/*
 *  Copyright (c) 2008, Atheros Communications Inc.  All rights reserved.
 * 
 */

#include "ath_internal.h"
#include "ath_antdiv.h"

#ifndef REMOVE_PKT_LOG
#include "pktlog.h"
extern struct ath_pktlog_funcs *g_pktlog_funcs;
#define MAX_LOG 128
#endif

void
ath_slow_ant_div_init(struct ath_antdiv *antdiv, struct ath_softc *sc, int32_t rssitrig,
                      u_int32_t min_dwell_time, u_int32_t settle_time)
{
    antdiv->antdiv_sc = sc;
    antdiv->antdiv_rssitrig = rssitrig;
    antdiv->antdiv_min_dwell_time = min_dwell_time * 1000;
    antdiv->antdiv_settle_time = settle_time * 1000;
}

void
ath_slow_ant_div_start(struct ath_antdiv *antdiv, u_int8_t num_antcfg, const u_int8_t *bssid)
{
    antdiv->antdiv_num_antcfg   = num_antcfg < ATH_ANT_DIV_MAX_CFG ? num_antcfg : ATH_ANT_DIV_MAX_CFG;
    antdiv->antdiv_state        = ATH_ANT_DIV_IDLE;
    antdiv->antdiv_curcfg       = 0;
    antdiv->antdiv_bestcfg      = 0;
    antdiv->antdiv_laststatetsf = 0;

    OS_MEMCPY(antdiv->antdiv_bssid, bssid, sizeof(antdiv->antdiv_bssid));
    
    antdiv->antdiv_start = 1;
}

void
ath_slow_ant_div_stop(struct ath_antdiv *antdiv)
{
    antdiv->antdiv_start = 0;
}

static void find_max_sdev(int8_t *val_in, u_int8_t num_val, u_int8_t *max_index, u_int16_t *sdev)
{
    int8_t   *max_val, *val = val_in, *val_last = val_in + num_val;

    if (val_in && num_val) {
        max_val = val;
        while (++val != val_last) {
            if (*val > *max_val)
                max_val = val;
        };

        if (max_index)
            *max_index = (u_int8_t)(max_val - val_in);

        if (sdev) {
            val = val_in;
            *sdev = 0;
            do {
                *sdev += (*max_val - *val);
            } while (++val != val_last);
        }
    }
}

void
ath_slow_ant_div(struct ath_antdiv *antdiv, struct ieee80211_frame *wh, struct ath_rx_status *rx_stats)
{
    struct ath_softc *sc = antdiv->antdiv_sc;
    struct ath_hal   *ah = sc->sc_ah;
    u_int64_t curtsf = 0;
    u_int8_t  bestcfg, bestcfg_chain[ATH_ANT_DIV_MAX_CHAIN], curcfg = antdiv->antdiv_curcfg;
    u_int16_t sdev_chain[ATH_ANT_DIV_MAX_CHAIN];
#ifndef REMOVE_PKT_LOG
    char logtext[MAX_LOG];
#endif

    if (antdiv->antdiv_start && IEEE80211_IS_BEACON(wh) && ATH_ADDR_EQ(wh->i_addr3, antdiv->antdiv_bssid)){
            antdiv->antdiv_lastbrssictl[0][curcfg] = rx_stats->rs_rssi_ctl0;
            antdiv->antdiv_lastbrssictl[1][curcfg] = rx_stats->rs_rssi_ctl1;
            antdiv->antdiv_lastbrssictl[2][curcfg] = rx_stats->rs_rssi_ctl2;
/* not yet
            antdiv->antdiv_lastbrssi[curcfg] = rx_stats->rs_rssi;
            antdiv->antdiv_lastbrssiext[0][curcfg] = rx_stats->rs_rssi_ext0;
            antdiv->antdiv_lastbrssiext[1][curcfg] = rx_stats->rs_rssi_ext1;
            antdiv->antdiv_lastbrssiext[2][curcfg] = rx_stats->rs_rssi_ext2;
*/
            antdiv->antdiv_lastbtsf[curcfg] = ath_hal_gettsf64(sc->sc_ah);
            curtsf = antdiv->antdiv_lastbtsf[curcfg];
    } else {
        return;
    }

    switch (antdiv->antdiv_state) {
    case ATH_ANT_DIV_IDLE:
        if ((curtsf - antdiv->antdiv_laststatetsf) > antdiv->antdiv_min_dwell_time) {
            int8_t min_rssi = antdiv->antdiv_lastbrssictl[0][curcfg];
            int      i;
            for (i = 1; i < ATH_ANT_DIV_MAX_CHAIN; i++) {
                if (antdiv->antdiv_lastbrssictl[i][curcfg] != (-128)) {
                    if (antdiv->antdiv_lastbrssictl[i][curcfg] < min_rssi) {
                        min_rssi = antdiv->antdiv_lastbrssictl[i][curcfg];
                    }
                }
            }
            if (min_rssi < antdiv->antdiv_rssitrig) {
                curcfg ++;
                if (curcfg == antdiv->antdiv_num_antcfg) {
                    curcfg = 0;
                }

                if (HAL_OK == ath_hal_selectAntConfig(ah, curcfg)) {
                    antdiv->antdiv_bestcfg = antdiv->antdiv_curcfg;
                    antdiv->antdiv_curcfg = curcfg;
                    antdiv->antdiv_laststatetsf = curtsf;
                    antdiv->antdiv_state = ATH_ANT_DIV_SCAN;
#ifndef REMOVE_PKT_LOG
                    /* do pktlog */
                    {
                        ath_sprintf(logtext, MAX_LOG, "SlowAntDiv: select_antcfg = %d\n", antdiv->antdiv_curcfg);
                        ath_log_text(sc, logtext, 0); 
                    }
#endif
                }
            }
        }
        break;
    
    case ATH_ANT_DIV_SCAN:
        if((curtsf - antdiv->antdiv_laststatetsf) < antdiv->antdiv_settle_time)
            break;

        curcfg ++;
        if (curcfg == antdiv->antdiv_num_antcfg) {
            curcfg = 0;
        }

        if (curcfg == antdiv->antdiv_bestcfg) {
            u_int16_t *max_sdev;
            int       i;
            for (i = 0; i < ATH_ANT_DIV_MAX_CHAIN; i++) {
                find_max_sdev(&antdiv->antdiv_lastbrssictl[i][0], antdiv->antdiv_num_antcfg, &bestcfg_chain[i], &sdev_chain[i]);
#ifndef REMOVE_PKT_LOG
                /* do pktlog */
                {
                    ath_sprintf(logtext, MAX_LOG, "SlowAntDiv: best cfg (chain%d) = %d (cfg0:%d, cfg1:%d, sdev:%d)\n", i, 
                            bestcfg_chain[i], antdiv->antdiv_lastbrssictl[i][0],
                            antdiv->antdiv_lastbrssictl[i][1], sdev_chain[i]);
                    ath_log_text(sc, logtext, 0); 
                }
#endif
            }

            max_sdev = sdev_chain;
            for (i = 1; i < ATH_ANT_DIV_MAX_CHAIN; i++) {
                if (*(sdev_chain + i) > *max_sdev) {
                    max_sdev = sdev_chain + i;
                }
            }
            bestcfg = bestcfg_chain[(max_sdev - sdev_chain)];

            if (bestcfg != antdiv->antdiv_curcfg) {
                if (HAL_OK == ath_hal_selectAntConfig(ah, bestcfg)) {
                    antdiv->antdiv_bestcfg = bestcfg;
                    antdiv->antdiv_curcfg = bestcfg;
#ifndef REMOVE_PKT_LOG
                    /* do pktlog */
                    {
                        ath_sprintf(logtext, MAX_LOG, "SlowAntDiv: select best cfg = %d\n", antdiv->antdiv_curcfg);
                        ath_log_text(sc, logtext, 0); 
                    }
#endif
                }
            }
            antdiv->antdiv_laststatetsf = curtsf;
            antdiv->antdiv_state = ATH_ANT_DIV_IDLE;

        } else {
            if (HAL_OK == ath_hal_selectAntConfig(ah, curcfg)) {
                antdiv->antdiv_curcfg = curcfg;
                antdiv->antdiv_laststatetsf = curtsf;
                antdiv->antdiv_state = ATH_ANT_DIV_SCAN;
#ifndef REMOVE_PKT_LOG
                /* do pktlog */
                {
                    ath_sprintf(logtext, MAX_LOG, "SlowAntDiv: select ant cfg = %d\n", antdiv->antdiv_curcfg);
                    ath_log_text(sc, logtext, 0); 
                }
#endif
            }
        }

        break;
    }
}
