/*
 *  Copyright (c) 2005, Atheros Communications Inc.  All Rights Reserved.
 *  This module contains an interface/shim layer between the HW layer and our 
 *  ath layer.  
 */

#include <osdep.h>
#include <wbuf.h>

#include "ath_internal.h"

#include <net80211/_ieee80211.h>

#ifndef EXPORT_SYMTAB
#define EXPORT_SYMTAB
#endif

/*
 * Indication of hardware PHY state change
 */
void
ath_hw_phystate_change(struct ath_softc *sc, int newstate)
{
    if (sc->sc_hw_phystate == newstate)
        return;

    sc->sc_hw_phystate = newstate;  /* save the hardware switch state */

#ifdef ATH_SUPPORT_STA
    /* TODO: rfkill support in Linux*/
    if (newstate)
    {	
        ath_radio_enable(sc);
        sc->sc_ieee_ops->ath_net80211_resume(sc->sc_ieee);
    }	
    else
    {
        sc->sc_ieee_ops->ath_net80211_suspend(sc->sc_ieee);
        ath_radio_disable(sc);
    }	
#endif
}

void ath_internal_reset(struct ath_softc *sc)
{
    ath_reset_start(sc, 0);
    ath_reset(sc);
    ath_reset_end(sc, 0);
}

void ath_handle_tx_intr(struct ath_softc *sc)
{
    ath_tx_tasklet(sc);
}

void ath_handle_rx_intr(struct ath_softc *sc)
{
    ath_rx_tasklet(sc, 0);
}

/*
 * Linux's specific rx indicator
 * 
 * In Linux, the wbuf indicated to upper stack won't be returned to us.
 * So we have to allocate a new one and queue it by ourselves.
 */
int ath_rx_indicate(struct ath_softc *sc, wbuf_t wbuf, ieee80211_rx_status_t *status, u_int16_t keyix)
{
    struct ath_buf *bf = ATH_GET_RX_CONTEXT_BUF(wbuf);
    wbuf_t nwbuf;
    int type;

    /* indicate frame to the stack, which will free the old wbuf. */
    type = sc->sc_ieee_ops->rx_indicate(sc->sc_ieee, wbuf, status, keyix);

    /* allocate a new wbuf and queue it to for H/W processing */
    nwbuf = ath_rxbuf_alloc(sc, sc->sc_rxbufsize);
    if (nwbuf != NULL) {
        bf->bf_mpdu = nwbuf;
        bf->bf_buf_addr = wbuf_map_single(sc->sc_osdev, nwbuf, BUS_DMA_FROMDEVICE,
                                          OS_GET_DMA_MEM_CONTEXT(bf, bf_dmacontext));
        ATH_SET_RX_CONTEXT_BUF(nwbuf, bf);

        /* queue the new wbuf to H/W */
        ath_rx_requeue(sc, nwbuf);
    }

    return type;
}


#ifdef ATH_SUPPORT_STA
#ifdef CONFIG_SYSCTL

enum {
        ATH_RADIO_ON    = 1,
};

static int
ATH_SYSCTL_DECL(ath_sysctl_halparam, ctl, write, filp, buffer, lenp, ppos)
{
        struct ath_softc *sc = ctl->extra1;
        static u_int val;
        int ret=0;

        ctl->data = &val;
        ctl->maxlen = sizeof(val);
        if (!write) {
		switch ((long)ctl->extra2) {
			case ATH_RADIO_ON:
				if (sc->sc_hw_phystate)
					val = 1;
				else
					val = 0;
				break;
			default:
				return -EINVAL;
		}
                ret = ATH_SYSCTL_PROC_DOINTVEC(ctl, write, filp, buffer,
                                lenp, ppos);
        }
	return ret;
}


/* sysctl entries: /proc/sys/dev/wifiN/... */
static const ctl_table ath_sysctl_template[] = {
        { .ctl_name     = CTL_AUTO,
          .procname     = "radio_on",
          .mode         = 0444,
          .proc_handler = ath_sysctl_halparam,
          .extra2       = (void *)ATH_RADIO_ON,
        },
	/* may add more in the future */
        { 0 }
};

void
ath_dynamic_sysctl_register(struct ath_softc *sc)
{
        int i, space;
	char *devname = sc->sc_osdev->netdev->name ;

        space = 5*sizeof(struct ctl_table) + sizeof(ath_sysctl_template);
        sc->sc_sysctls = kmalloc(space, GFP_KERNEL);
        if (sc->sc_sysctls == NULL) {
                printk("%s: no memory for sysctl table!\n", __func__);
                return;
        }

        /* setup the table */
        memset(sc->sc_sysctls, 0, space);
        sc->sc_sysctls[0].ctl_name = CTL_DEV;
        sc->sc_sysctls[0].procname = "dev";
        sc->sc_sysctls[0].mode = 0555;
        sc->sc_sysctls[0].child = &sc->sc_sysctls[2];
        /* [1] is NULL terminator */
        sc->sc_sysctls[2].ctl_name = CTL_AUTO;
        sc->sc_sysctls[2].procname = devname ;
        sc->sc_sysctls[2].mode = 0555;
        sc->sc_sysctls[2].child = &sc->sc_sysctls[4];
        /* [3] is NULL terminator */
        /* copy in pre-defined data */
        memcpy(&sc->sc_sysctls[4], ath_sysctl_template,
                sizeof(ath_sysctl_template));

        /* add in dynamic data references */
        for (i = 4; sc->sc_sysctls[i].procname; i++)
                if (sc->sc_sysctls[i].extra1 == NULL)
                        sc->sc_sysctls[i].extra1 = sc;

        /* and register everything */
#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,20)
        sc->sc_sysctl_header = register_sysctl_table(sc->sc_sysctls);
#else
        sc->sc_sysctl_header = register_sysctl_table(sc->sc_sysctls, 1);
#endif
        if (!sc->sc_sysctl_header) {
                printk("%s: failed to register sysctls!\n", devname);
                kfree(sc->sc_sysctls);
                sc->sc_sysctls = NULL;
        }
}

void
ath_dynamic_sysctl_unregister(struct ath_softc *sc)
{
        if (sc->sc_sysctl_header) {
                unregister_sysctl_table(sc->sc_sysctl_header);
                sc->sc_sysctl_header = NULL;
        }
        if (sc->sc_sysctls) {
                kfree(sc->sc_sysctls);
                sc->sc_sysctls = NULL;
        }
}

EXPORT_SYMBOL(ath_dynamic_sysctl_register);
EXPORT_SYMBOL(ath_dynamic_sysctl_unregister);

#endif /* CONFIG_SYSCTL */
#endif /* ATH_SUPPORT_STA */


/*
 * Linux module glue
 */
static char *dev_info = "ath_dev";

MODULE_AUTHOR("Atheros Communications, Inc.");
MODULE_DESCRIPTION("Atheros Device Module");
#ifdef MODULE_LICENSE
MODULE_LICENSE("Proprietary");
#endif

static int __init
init_ath_dev(void)
{
    /* XXX version is a guess; where should it come from? */
    printk(KERN_INFO "%s: "
           "Copyright (c) 2001-2007 Atheros Communications, Inc, "
           "All Rights Reserved\n", dev_info);

    return 0;
}
module_init(init_ath_dev);

static void __exit
exit_ath_dev(void)
{
    printk(KERN_INFO "%s: driver unloaded\n", dev_info);
}
module_exit(exit_ath_dev);

EXPORT_SYMBOL(ath_dev_attach);
EXPORT_SYMBOL(ath_dev_free);
EXPORT_SYMBOL(ath_initialize_timer);
EXPORT_SYMBOL(ath_cancel_timer);
EXPORT_SYMBOL(ath_start_timer);
EXPORT_SYMBOL(ath_timer_is_active);

#ifndef REMOVE_PKT_LOG
#include "pktlog.h"
extern struct ath_pktlog_funcs *g_pktlog_funcs;
EXPORT_SYMBOL(g_pktlog_funcs);
#endif
