/*
 * Copyright (c) 2002-2005 Sam Leffler, Errno Consulting
 * Copyright (c) 2002-2005 Atheros Communications, Inc.
 * All rights reserved.
 *
 */
#include "opt_ah.h"

#ifdef AH_SUPPORT_AR5416

#include "ah.h"
#include "ah_internal.h"

#include "ar5416/ar5416.h"
#include "ar5416/ar5416reg.h"
#include "ar5416/ar5416phy.h"


/*
 * Checks to see if an interrupt is pending on our NIC
 *
 * Returns: TRUE    if an interrupt is pending
 *          FALSE   if not
 */
HAL_BOOL
ar5416IsInterruptPending(struct ath_hal *ah)
{
#ifndef AR9100
    /*
     * Some platforms trigger our ISR before applying power to
     * the card, so make sure.
     */
    u_int32_t host_isr = OS_REG_READ(ah, AR_INTR_ASYNC_CAUSE);
    if ((host_isr & AR_INTR_ASYNC_USED) && (host_isr != AR_INTR_SPURIOUS))
        return AH_TRUE;

    host_isr = OS_REG_READ(ah, AR_INTR_SYNC_CAUSE);
    if ((host_isr & (AR_INTR_SYNC_DEFAULT | AR_INTR_SYNC_MASK_GPIO)) && (host_isr != AR_INTR_SPURIOUS))
        return AH_TRUE;

    return AH_FALSE;
#else
    return AH_TRUE;
#endif
}

/*
 * Reads the Interrupt Status Register value from the NIC, thus deasserting
 * the interrupt line, and returns both the masked and unmasked mapped ISR
 * values.  The value returned is mapped to abstract the hw-specific bit
 * locations in the Interrupt Status Register.
 *
 * Returns: A hardware-abstracted bitmap of all non-masked-out
 *          interrupts pending, as well as an unmasked value
 */
HAL_BOOL
ar5416GetPendingInterrupts(struct ath_hal *ah, HAL_INT *masked)
{
    u_int32_t isr = 0;
    u_int32_t mask2=0;
#ifndef AR9100
    HAL_BOOL fatal_int = AH_FALSE;
    u_int32_t sync_cause = 0;
    u_int32_t async_cause;
    HAL_CAPABILITIES *pCap = &AH_PRIVATE(ah)->ah_caps;

    /* Make sure mac interrupt is pending in async interrupt cause register */
    async_cause = OS_REG_READ(ah, AR_INTR_ASYNC_CAUSE);
    if (async_cause & AR_INTR_ASYNC_USED) {
        /* RTC may not be on since it runs on a slow 32khz clock so check its status to be sure */
        if ((OS_REG_READ(ah, AR_RTC_STATUS) & AR_RTC_STATUS_M) == AR_RTC_STATUS_ON) {
            isr = OS_REG_READ(ah, AR_ISR);
        }
    }
    
    sync_cause = OS_REG_READ(ah, AR_INTR_SYNC_CAUSE) & (AR_INTR_SYNC_DEFAULT | AR_INTR_SYNC_MASK_GPIO);

    *masked = 0;

    if (!isr && !sync_cause)
        return AH_FALSE;

#else
    *masked = 0;
    isr = OS_REG_READ(ah, AR_ISR);
#endif

    if (isr) {
        struct ath_hal_5416 *ahp = AH5416(ah);

        if (isr & AR_ISR_BCNMISC) {
            u_int32_t isr2;
            isr2 = OS_REG_READ(ah, AR_ISR_S2);
            if (isr2 & AR_ISR_S2_TIM)
                mask2 |= HAL_INT_TIM;
            if (isr2 & AR_ISR_S2_DTIM)
                mask2 |= HAL_INT_DTIM;
            if (isr2 & AR_ISR_S2_DTIMSYNC)
                mask2 |= HAL_INT_DTIMSYNC;
            if (isr2 & (AR_ISR_S2_CABEND ))
                mask2 |= HAL_INT_CABEND;
            if (isr2 & AR_ISR_S2_GTT)
                mask2 |= HAL_INT_GTT;
            if (isr2 & AR_ISR_S2_CST)
                mask2 |= HAL_INT_CST;
            if (isr2 & AR_ISR_S2_TSFOOR)
                mask2 |= HAL_INT_TSFOOR;
        }

        isr = OS_REG_READ(ah, AR_ISR_RAC);
        if (isr == 0xffffffff) {
            *masked = 0;
            return AH_FALSE;
        }

        *masked = isr & HAL_INT_COMMON;

        /*
         * When interrupt mitigation is switched on, we fake a normal RX or TX
         * interrupt when we received a mitigated interrupt. This way, the upper
         * layer do not need to know about feature.
         */
        if (ahp->ah_intrMitigation) {
            /* Only Rx interrupt mitigation. No Tx intr. mitigation. */
            if (isr & (AR_ISR_RXMINTR | AR_ISR_RXINTM)) {
                *masked |= HAL_INT_RX;
            }
        }

        if (isr & AR_ISR_GENTMR) {
            u_int32_t s5_s;
            s5_s = OS_REG_READ(ah, AR_ISR_S5_S);
            HDPRINTF(ah, HAL_DBG_INTERRUPT,
                "%s: GENTIMER, ISR_RAC=0x%x ISR_S2_S=0x%x\n", __func__,
                isr, s5_s);
            if (s5_s & AR_ISR_S5_GENTIMER7) {
                *masked |= HAL_INT_GENTIMER;
            }
        }

        if (isr & (AR_ISR_RXOK | AR_ISR_RXERR))
            *masked |= HAL_INT_RX;
        if (isr & (AR_ISR_TXOK | AR_ISR_TXDESC | AR_ISR_TXERR | AR_ISR_TXEOL)) {
            u_int32_t           s0_s, s1_s;

            *masked |= HAL_INT_TX;

            s0_s = OS_REG_READ(ah, AR_ISR_S0_S);
            ahp->ah_intrTxqs |= MS(s0_s, AR_ISR_S0_QCU_TXOK);
            ahp->ah_intrTxqs |= MS(s0_s, AR_ISR_S0_QCU_TXDESC);

            s1_s = OS_REG_READ(ah, AR_ISR_S1_S);
            ahp->ah_intrTxqs |= MS(s1_s, AR_ISR_S1_QCU_TXERR);
            ahp->ah_intrTxqs |= MS(s1_s, AR_ISR_S1_QCU_TXEOL);
        }

        /*
         * Do not treat receive overflows as fatal for owl.
         */
        if (isr & AR_ISR_RXORN) {
            HDPRINTF(ah, HAL_DBG_INTERRUPT, "%s: receive FIFO overrun interrupt\n", __func__);
            // *masked |= HAL_INT_FATAL;
        }

#ifndef AR9100
        if (! pCap->halAutoSleepSupport) {
            u_int32_t isr5 = OS_REG_READ(ah, AR_ISR_S5_S);
            if (isr5 & AR_ISR_S5_TIM_TIMER) {
                *masked |= HAL_INT_TIM_TIMER;
            }
        }
#endif

        *masked |= mask2;
    }

#ifndef AR9100
    if (async_cause & AR_INTR_ASYNC_CAUSE_GPIO)
        *masked |= HAL_INT_GPIO;

    if (sync_cause) {
        fatal_int = (sync_cause & (AR_INTR_SYNC_HOST1_FATAL | AR_INTR_SYNC_HOST1_PERR)) ?
                    AH_TRUE : AH_FALSE;

        if (AH_TRUE == fatal_int) {
            if (sync_cause & AR_INTR_SYNC_HOST1_FATAL) {
                HDPRINTF(ah, HAL_DBG_UNMASKABLE, "%s: received PCI FATAL interrupt\n", __func__);
            }
            if (sync_cause & AR_INTR_SYNC_HOST1_PERR) {
                HDPRINTF(ah, HAL_DBG_UNMASKABLE, "%s: received PCI PERR interrupt\n", __func__);
            }

            /* Can mark this as fatal when upper layer does resets for
             * fatal interrupts. Otherwise, this can be uncommented to aid
             * in debugging PCI errors.
             */
            // *masked |= HAL_INT_FATAL;
        }
        if (sync_cause & AR_INTR_SYNC_RADM_CPL_TIMEOUT) {
            HDPRINTF(ah, HAL_DBG_INTERRUPT, 
                "%s: AR_INTR_SYNC_RADM_CPL_TIMEOUT\n",
                __func__);
            
            OS_REG_WRITE(ah, AR_RC, AR_RC_HOSTIF);
            OS_REG_WRITE(ah, AR_RC, 0);
            *masked |= HAL_INT_FATAL;
        }
        if (sync_cause & AR_INTR_SYNC_LOCAL_TIMEOUT) {
            HDPRINTF(ah, HAL_DBG_INTERRUPT, 
                "%s: AR_INTR_SYNC_LOCAL_TIMEOUT\n",
                __func__);
        }

        if (sync_cause & AR_INTR_SYNC_MASK_GPIO) {
            *masked |= HAL_INT_GPIO;
            HDPRINTF(ah, HAL_DBG_INTERRUPT, "%s: AR_INTR_SYNC_GPIO\n", __func__);
        }

        OS_REG_WRITE(ah, AR_INTR_SYNC_CAUSE_CLR, sync_cause);
        // Flush prior write
        (void) OS_REG_READ(ah, AR_INTR_SYNC_CAUSE_CLR);
    }
#endif

    return AH_TRUE;
}

HAL_INT
ar5416GetInterrupts(struct ath_hal *ah)
{
    return AH5416(ah)->ah_maskReg;
}

/*
 * Atomically enables NIC interrupts.  Interrupts are passed in
 * via the enumerated bitmask in ints.
 */
HAL_INT
ar5416SetInterrupts(struct ath_hal *ah, HAL_INT ints)
{
    struct ath_hal_5416 *ahp = AH5416(ah);
    u_int32_t omask = ahp->ah_maskReg;
    u_int32_t mask, mask2;
    HAL_CAPABILITIES *pCap = &AH_PRIVATE(ah)->ah_caps;

    HDPRINTF(ah, HAL_DBG_INTERRUPT, "%s: 0x%x => 0x%x\n", __func__, omask, ints);

    if (omask & HAL_INT_GLOBAL) {
        HDPRINTF(ah, HAL_DBG_INTERRUPT, "%s: disable IER\n", __func__);
        OS_REG_WRITE(ah, AR_IER, AR_IER_DISABLE);
        (void) OS_REG_READ(ah, AR_IER);   /* flush write to HW */      
#ifndef AR9100
        OS_REG_WRITE(ah, AR_INTR_ASYNC_ENABLE, 0);
        (void) OS_REG_READ(ah, AR_INTR_ASYNC_ENABLE); /* flush write to HW */

        OS_REG_WRITE(ah, AR_INTR_SYNC_ENABLE, 0);
        (void) OS_REG_READ(ah, AR_INTR_SYNC_ENABLE); /* flush write to HW */

#endif
    }

    mask = (ints & HAL_INT_COMMON) & ~HAL_INT_GPIO; /* bit 24 is reserved */;
    mask2 = 0;
    
    if (ints & HAL_INT_TX) {
        if (ahp->ah_txOkInterruptMask)
            mask |= AR_IMR_TXOK;
        if (ahp->ah_txDescInterruptMask)
            mask |= AR_IMR_TXDESC;
        if (ahp->ah_txErrInterruptMask)
            mask |= AR_IMR_TXERR;
        if (ahp->ah_txEolInterruptMask)
            mask |= AR_IMR_TXEOL;
    }
    if (ints & HAL_INT_RX) {
        mask |= AR_IMR_RXERR;
        if (ahp->ah_intrMitigation) {
            /* Only Rx interrupt mitigation. No Tx intr. mitigation. */
            mask |=  AR_IMR_RXMINTR | AR_IMR_RXINTM;
        } else {
            mask |= AR_IMR_RXOK | AR_IMR_RXDESC;
        }
        if (! pCap->halAutoSleepSupport) {
            mask |= AR_IMR_GENTMR;
        }
    }

    if (ints & (HAL_INT_BMISC)) {
        mask |= AR_IMR_BCNMISC;
        if (ints & HAL_INT_TIM)
            mask2 |= AR_IMR_S2_TIM;
        if (ints & HAL_INT_DTIM)
            mask2 |= AR_IMR_S2_DTIM;
        if (ints & HAL_INT_DTIMSYNC)
            mask2 |= AR_IMR_S2_DTIMSYNC;
        if (ints & HAL_INT_CABEND)
            mask2 |= AR_IMR_S2_CABEND;
        if (ints & HAL_INT_TSFOOR)
            mask2 |= AR_IMR_S2_TSFOOR;
    }
    
    if (ints & (HAL_INT_GTT | HAL_INT_CST)) {
        mask |= AR_IMR_BCNMISC;
        if (ints & HAL_INT_GTT)
            mask2 |= AR_IMR_S2_GTT;
        if (ints & HAL_INT_CST)
            mask2 |= AR_IMR_S2_CST;
    }

   if (ints & HAL_INT_GENTIMER) {
        HDPRINTF(ah, HAL_DBG_INTERRUPT, "%s: enabling gen timer\n", __func__);
        OS_REG_WRITE(
            ah, AR_IMR_S5, OS_REG_READ(ah, AR_IMR_S5) | AR_IMR_S5_GENTIMER7);
        mask |= AR_IMR_GENTMR;
    }

    /* Write the new IMR and store off our SW copy. */
    HDPRINTF(ah, HAL_DBG_INTERRUPT, "%s: new IMR 0x%x\n", __func__, mask);
    OS_REG_WRITE(ah, AR_IMR, mask);
    /* For Some processors: It was found with the serial driver that,
       under certain conditions, a spurious CIC interrupt could be triggered
       if the interrupt enable/mask registers weren't updated before the
       CIC mask register.  A read of the register just written to
       enable/disable such interrupts should remedy the problem.  It was
       suggested that the madwifi driver was also succeptible to these
       spurious CIC interrupts, so a similar fix may be needed.
    */
    (void) OS_REG_READ(ah, AR_IMR);

    mask = OS_REG_READ(ah, AR_IMR_S2) & ~(AR_IMR_S2_TIM |
                    AR_IMR_S2_DTIM |
                    AR_IMR_S2_DTIMSYNC |
                    AR_IMR_S2_CABEND |
                    AR_IMR_S2_CABTO  |
                    AR_IMR_S2_TSFOOR |
                    AR_IMR_S2_GTT |
                    AR_IMR_S2_CST );
    OS_REG_WRITE(ah, AR_IMR_S2, mask | mask2 );
    ahp->ah_maskReg = ints;

    if (! pCap->halAutoSleepSupport) {
        if (ints & HAL_INT_TIM_TIMER) {
            OS_REG_SET_BIT(ah, AR_IMR_S5, AR_IMR_S5_TIM_TIMER);
        }
        else {
            OS_REG_CLR_BIT(ah, AR_IMR_S5, AR_IMR_S5_TIM_TIMER);
        }
    }

    /* Re-enable interrupts if they were enabled before. */
    if (ints & HAL_INT_GLOBAL) {
        HDPRINTF(ah, HAL_DBG_INTERRUPT, "%s: enable IER\n", __func__);
        OS_REG_WRITE(ah, AR_IER, AR_IER_ENABLE);
        /* See explanation above... */
        (void) OS_REG_READ(ah, AR_IER);
#ifndef AR9100
        mask = AR_INTR_MAC_IRQ;
        if (ints & HAL_INT_GPIO)
            mask |= SM(ahp->ah_gpioMask, AR_INTR_ASYNC_MASK_GPIO);
        OS_REG_WRITE(ah, AR_INTR_ASYNC_ENABLE, mask);
        OS_REG_WRITE(ah, AR_INTR_ASYNC_MASK, mask);

        /*
         * debug - enable to see all synchronous interrupts status
         * Enable synchronous GPIO interrupts as well, since some async GPIO interrupts
         * don't wake the chip up.
         */
        mask = 0;
        if (ints & HAL_INT_GPIO)
            mask |= SM(ahp->ah_gpioMask, AR_INTR_SYNC_MASK_GPIO);
        OS_REG_WRITE(ah, AR_INTR_SYNC_ENABLE, (AR_INTR_SYNC_DEFAULT | mask));
        OS_REG_WRITE(ah, AR_INTR_SYNC_MASK, (AR_INTR_SYNC_DEFAULT | mask));

#endif
        HDPRINTF(ah,  HAL_DBG_INTERRUPT, "AR_IMR 0x%x IER 0x%x\n", OS_REG_READ(ah, AR_IMR), OS_REG_READ(ah, AR_IER));
    }

    return omask;
}
#endif /* AH_SUPPORT_AR5416 */

