/*
 *  Copyright (c) 2008 Atheros Communications Inc.  All rights reserved.
 */

#ifndef _IEEE80211_API_H_
#define _IEEE80211_API_H_

/*
 * Public Interface for Atheros Upper MAC Layer
 */

#include <osdep.h>
#include <wbuf.h>
#include <_ieee80211.h>
#include <ieee80211.h>
#include <ieee80211_defines.h>

/* @get the list of vaps. 
 *  ARGS:
 *   devhandle  : handle to the dev.   
 *   vaps       : vaps array (caller owns memory). 
 *   len        : length of the vaps array. 
 *   
 *  RETURNS:
 *        number of elements retured in the vaps array. 
 */
u_int32_t wlan_get_vaps(wlan_dev_t devhandle, wlan_if_t *vaps, u_int32_t len);

/*
 * @returns the HW capabilities 
 *  ARGS:
 *       devhandle : device handle
 */
u_int32_t wlan_get_HWcapabilities(wlan_dev_t devhandle, ieee80211_cap cap);

/*
 * @set simple configuration parameter. 
 * 
 * ARGS :
 *  wlan_if_t       : handle to the vap.
 *  param           : simple config paramaeter.
 *  val             : value of the parameter.
 *  RETURNS: 0  on success and -ve on failure.
 */
int wlan_set_device_param(wlan_dev_t devhandle, ieee80211_device_param param, u_int32_t val);

/*
 * @get simple configuration parameter. 
 * 
 * ARGS :
 *  wlan_if_t       : handle to the vap.
 *  param           : simple config paramaeter.
 *  RETURNS:value of the parameter. 
 */
u_int32_t wlan_get_device_param(wlan_dev_t devhandle, ieee80211_device_param param);

/**
 * @creates a new vap with the given mode and flags.
 * ARGS :
 *  wlan_dev_t    : handle to the radio .   
 *  wlan_opmode   : opmode of the vap
 *  flags         : flags as defined above.
 *  bssid         : bssid of the new vap created. only valid if opmode is IEEE80211_M_WDS.
 *
 * RETURNS:
 *  on success return an opaque handle to newly created vap.
 *  on failure returns NULL.
 */
wlan_if_t wlan_vap_create(wlan_dev_t devhandle,
                          enum ieee80211_opmode opmode, int flags, u_int8_t *bssid);

/**
 * @delete a vap. 
 * ARGS :
 *  ieee80211_if_t : handle to the vap .   
 *
 * RETURNS:
 *  on success returns 0. 
 *  on failure returns a negative value.
 */
int wlan_vap_delete(wlan_if_t vaphandle); 

/**
 * @register event handler table with a vap.
 * ARGS :
 *  ieee80211_if_t          : handle to the vap .   
 *  os_if_t                 : handle to os network interface. opaque to the implementor.   
 *  ieee80211_event_handler : table of handlers to receive 802.11 events.
 *                            if table is NULL it unregisters the previously registered
 *                            handlers. returns an error if handler table is registered    
 *                            already. Note that the caller owns the memory of the even table structure.
 *                            caller should not deallocate memory while it is registered wit the vap. 
 * RETURNS:
 *  on success return 0.
 *  on failure returns -1.
 */
int wlan_vap_register_event_handlers(wlan_if_t vaphandle,
                                     os_if_t osif,
                                     wlan_event_handler_table *evtable);

/* @get the os_if_t handle registered using the
 *    wlan_vap_register_event_handlers function. 
 *  
 *  ARGS:
 *   wlan_if_t  : handle to the vap .   
 *  RETURNS:
 *        dev_handle. 
 */
os_if_t wlan_vap_get_registered_handle(wlan_if_t vaphandle);

/**
 * @register mlme event handler table with a vap.
 * ARGS :
 *  wlan_if_t               : handle to the vap .   
 *  os_if_t                 : handle opaque to the implementor.   
 *  wlan_mlme_event_handler : table of handlers to receive 802.11 mlme events.
 *                            if table is NULL it unregisters the previously registered
 *                            handlers. returns an error if handler table is registered    
 *                            already. Note that the caller owns the memory of the even table structure.
 *                            caller should not deallocate memory while it is registered wit the vap. 
 * RETURNS:
 *  on success return 0.
 *  on failure returns -1.
 */
int wlan_vap_register_mlme_event_handlers(wlan_if_t vaphandle,
                                          os_if_t,     
                                          wlan_mlme_event_handler_table *evtable);

/* @get the os_if_t handle registered using the
 *    wlan_vap_register_mlme_event_handlers function. 
 *  
 *  ARGS:
 *   wlan_if_t  : handle to the vap .   
 *  RETURNS:
 *        dev_handle. 
 */
os_if_t wlan_vap_get_mlme_registered_handle(wlan_if_t vaphandle);

/**
 * @register mmisc event handler table with a vap.
 * ARGS :
 *  wlan_if_t               : handle to the vap .   
 *  os_if_t                 : handle opaque to the implementor.   
 *  wlan_mlme_event_handler : table of misc handlers to receive 802.11 misc events.
 *                            returns an error if handler table is registered    
 *                            already. Note that the caller owns the memory of the even table structure.
 *                            caller should not deallocate memory while it is registered wit the vap. 
 * This API allows multiple modules to register thier own event handler tables.
 * RETURNS:
 *  on success return 0.
 *  on failure returns -1.
 */
int wlan_vap_register_misc_event_handlers(wlan_if_t vaphandle,
                                          os_if_t,     
                                          wlan_misc_event_handler_table *evtable);

/**
 * @unregister a misc event handler. 
 * ARGS :
 *  wlan_if_t                    : handle to the vap .
 *  os_if_t                      : handle opaque to the implementor.   
 *  wlan_mlme_event_handler      : table of misc handlers to be unregistred.
 * RETURNS:
 *  on success returns 0. 
 *  on failure returns a negative value.
 */
int wlan_vap_unregister_misc_event_handlers(wlan_if_t  vaphandle, os_if_t, 
                                       wlan_misc_event_handler_table *evtable);


/**
 * @start a vap. 
 * ARGS :
 *  ieee80211_if_t : handle to the vap .   
 *
 * RETURNS:
 *  on success returns 0. 
 *  on failure returns a negative value.
 */
int wlan_vap_start(wlan_if_t vaphandle); 

/**
 * @stop a vap. 
 * ARGS :
 *  ieee80211_if_t : handle to the vap .   
 *
 * RETURNS:
 *  on success returns 0. 
 *  on failure returns a negative value.
 */
int wlan_vap_stop(wlan_if_t vaphandle); 

/**
 * @send a data packet on vap. 
 * ARGS:
 *  ieee80211_if_t : handle to the vap .   
 *  wbuf_t         : wbuf to send.
 *
 * RETURNS:
 *  on success returns 0. 
 *  on failure returns a negative value.
 *  in case of failure it is the responsibility of the caller to free the wbuf.
 *  
 */
int wlan_vap_send(wlan_if_t vaphandle, wbuf_t wbuf); 

/**
 * @dump all the nodes allocated (useful for debugging). 
 * ARGS :
 *  wlan_dev_t                   : dev handle to the com  device .
 */    
void wlan_dump_alloc_nodes(wlan_dev_t devhandle);

/**
 * @get a unique scan requestor id. 
 * ARGS :
 *  wlan_if_t                : handle to the vap .
 *  module_name              : name of the module (useful for debugging). 
 *  req_id                   : param to return the uique requestor id.
 * RETURNS:
 *  on success returns 0. 
 *  on failure returns a negative value.
 */
int wlan_scan_get_requestor_id( wlan_if_t   vaphandle,
                                u_int8_t *module_name,
                                u_int16_t *req_id);

/**
 * @clear allocated scan requestor id.
 * ARGS :
 *  wlan_if_t                : handle to the vap.
 *  req_id                   : requestor id to be cleared.
 */
void wlan_scan_clear_requestor_id( wlan_if_t vaphandle, u_int16_t req_id);

/**
 * @get a name from requestor id. 
 * ARGS :
 *  wlan_if_t                : handle to the vap .
 *  req_id                   : requestor id.
 * RETURNS:
 *  on success returns pointer to module na,e. 
 *  on failure returns NULL.
 */
u_int8_t *wlan_scan_requestor_name(wlan_if_t vaphandle,  u_int16_t req_id);


/**
 * @start a scan . 
 * ARGS :
 *  wlan_if_t                    : handle to the vap .
 *  ieee80211_scan_params        : scan params.
 *  req_id                       : value to identify the module requesting
 *                                 scan. the value is the requestor id obtained from. 
 *                                 from wlan_scan_get_requestor id.
 *    
 * if there are multiple modules that can request scan then to implement policy on which
 * module has more priority than the other we need to identify which module issued the scan.
 * theo originator identifies originator of the scan. for example if a scan is in progress 
 * requested by  module A (say periodic BG scan) and if Module B( OS on behalf of user ) then
 * a typical  policy  might be to cancel the  BG scan issued by module A and start  a new scan    
 * from module B. To implement such a policy the policy implementor needs to know  the originator
 * of that scan in progress.
 * if number of channels passed in is 0 then  scanner will scan all the supported channels. 
 * note that in this case the scanner will still honor the 5GHZ,2GHZ , ACTIVE and passive flags. 
 *    
 *
 * RETURNS:
 *  on success returns 0. 
 *  on failure returns a negative value.
 *  if succesful
 *  IEEE80211_SCAN_START event is delivered via the event handler. 
 *  at the end of the scan IEEE80211_SCAN_DONE event is delivered 
 *  via the event handler registered. 
 *  if BSSID list is specified then IEEE80211_SCAN_BSSID_MATCH event  is delivered when 
 *  ever a new scan entry is added/updated that matches one of the BSSIDs in the list. 
 *  
 */
int wlan_scan_start(wlan_if_t             vaphandle,
                    ieee80211_scan_params *params, 
                    u_int16_t             req_id);

/**
 * @register a scan event handler. 
 * ARGS :
 *  wlan_if_t                    : handle to the vap .
 *  ieee80211_scan_event_handler : scan event handler 
 *  arg                          : argument passed back via the evnt handler
 * RETURNS:
 *  on success returns 0. 
 *  on failure returns a negative value.
 * allows more than one event handler to be registered. 
 */
int wlan_scan_register_event_handler(wlan_if_t                    vaphandle, 
                                     ieee80211_scan_event_handler evhandler, 
                                     void                         *arg);

/**
 * @unregister a scan event handler. 
 * ARGS :
 *  wlan_if_t                    : handle to the vap .
 *  ieee80211_scan_event_handler : scan event handler 
 * RETURNS:
 *  on success returns 0. 
 *  on failure returns a negative value.
 */
int wlan_scan_unregister_event_handler(wlan_if_t                    vaphandle, 
                                       ieee80211_scan_event_handler evhandler,
                                       void                         *arg);

/**
 * @cancel scan. 
 * ARGS :
 *  wlan_if_t                    : handle to the vap .
 *  req_id                       : id of the requestor
 *  flag_sync                    : indicates whether scan should terminate synchronously
 *
 * RETURNS:
 *  on success returns 0. 
 *  on failure returns a negative value.
 *
 *  when succesfully cancelled the scan, IEEE80211_SCAN_CANCELLED 
 *  event is delivered via the event handler registered
 *  event can be delivered in the context of the caller of the thread.
 *  
 *  if flag_sync is set, this function will sleep until scan completes.
 */
int wlan_scan_cancel(wlan_if_t vaphandle, u_int16_t req_id, bool flag_sync);

/**
 * @pause scan on vap. 
 * ARGS :
 *  wlan_if_t                    : handle to the vap .
 *  duration_ms                  : pause duration in ms
 *  channel                      : the channel where to pause
 *  req_id                       : id of the requestor
 *
 * RETURNS:
 *  on success returns 0. 
 *  on failure returns a negative value.
 *  If scan is succesfully paused, event IEEE80211_SCAN_PAUSED is delivered 
 *  to the event handler registered. 
 */
int wlan_scan_pause(wlan_if_t vaphandle,
                    u_int16_t duration_ms,
                    u_int16_t channel,
                    u_int16_t req_id);

/**
 * @resume scan on radio. 
 * ARGS :
 *  wlan_if_t                    : handle to the vap .
 *  req_id                       : id of the requestor
 *
 * RETURNS:
 *  on success returns 0. 
 *  on failure returns a negative value.
 *  If scan is succesfully paused, event IEEE80211_SCAN_RESUMED is delivered 
 *  to the event handler registered. 
 */
int wlan_scan_resume(wlan_if_t vaphandle, u_int16_t req_id);

/**
 * @returns the time when the last scan started.
 * ARGS :
 *  wlan_if_t            : handle to the vap .
 *                         then implementation would fill the passed scan info structure. 
 * RETURNS:
 *  returns the time when the last scan started.
 */
systime_t wlan_get_last_scan_time(wlan_if_t vaphandle);

/**
 * @returns info about last scan.
 * ARGS :
 *  wlan_if_t            : handle to the vap .
 *  info                 : if this is non NULL and if there was a last scan 
 *                         then implementation would fill the passed scan info structure. 
 * RETURNS:
 *  returns EOK if there was a scan before.
 *  returns ENXIO if there was no scan before - some returned data may not apply.
 */
int wlan_get_last_scan_info(wlan_if_t           vaphandle, 
                            ieee80211_scan_info *info);

/**
 * @indicates whether hw can transmit frames (i.e. we are in the home channel, 
 *  and not about to switch channels) .
 * ARGS :
 *  wlan_if_t            : handle to the vap .
 * RETURNS:
 *  returns 1 if data can be transmitted.
 *  returns 0 otherwise.
 */
int wlan_scan_can_transmit(wlan_if_t vaphandle);

/**
 * @indicates whether scan is in progress.
 * ARGS :
 *  wlan_if_t            : handle to the vap .
 * RETURNS:
 *  returns 1 if scan in progress.
 *  returns 0 otherwise.
 */
int wlan_scan_in_progress(wlan_if_t vaphandle);

/**
 * @notifies the scanner that connect has been lost.
 * ARGS :
 *  wlan_if_t            : handle to the vap .
 * RETURNS:
 *  none
 */
void wlan_scan_connection_lost(wlan_if_t vaphandle);

/**
 * @provides the default scan parameters .
 * ARGS :
 *  wlan_if_t            : handle to the vap .
 *  scan_params          : the data structure to be populated with the default values.
 *  opmode               : operation mode - affects some of the default values.
 *  active_scan_flag     : flag indicating whether the defaults should be for an active scan
 *  forced_scan_flag     : flag indicating whether the defaults should be for a forced scan
 *  connected_flag       : flag indicating connection state. 
 *  num_ssid             : number of desired SSIDs specified by the OS
 *  ssid_list            : list of desired SSIDs specified by the OS
 *  peer_count           : IBSS only - number of peers currently in the adhoc network.
 * RETURNS:
 *  none
 */
void wlan_set_default_scan_parameters(wlan_if_t             vaphandle, 
                                      ieee80211_scan_params *scan_params, 
                                      enum ieee80211_opmode opmode,
                                      bool                  active_scan_flag, 
                                      bool                  forced_scan_flag,
                                      bool                  connected_flag,
                                      u_int32_t             num_ssid,
                                      ieee80211_ssid        *ssid_list,
                                      int                   peer_count);

/**
 * @sets the scan priority.
 * ARGS :
 *  wlan_if_t            : handle to the vap .
 *  priority             : IEEE80211_SCAN_PRIORITY_HIGH indicates the scan should proceed even
 *                         in the presence of traffic;
 *                         IEEE80211_SCAN_PRIORITY_LOW indicates the scan should proceed
 *                         when no data traffic is present.
 * RETURNS:
 *  returns 1 if scan in progress and priority was set.
 *  returns 0 otherwise.
 */
int wlan_scan_set_priority(wlan_if_t vaphandle, ieee80211_scan_priority scanPriority);

typedef int (* ieee80211_scan_iter_func) (void *, wlan_scan_entry_t);

/**
 * @retrieve scan results from scan cache by iterating through scancache. 
 * ARGS :
 *  ieee80211_dev_t              : handle to the vap .   
 *  ieee80211_scanresult_handler : callback to deliver one scan entry  
 *
 * RETURNS:
 *  on success returns 0. 
 *  on failure returns a negative value.
 *  
 */
int wlan_scan_table_iterate(wlan_dev_t devhandle, ieee80211_scan_iter_func shandler, void *arg);

/**
 * @flush scan cache. 
 * ARGS :
 *  ieee80211_dev_t       : handle to the radio .   
 *
 */
void wlan_scan_table_flush(wlan_dev_t devhandle);

/*
 *  
 * @lock access to scan entry
 * ARGS :
 *  wlan_scan_entry_t        : handle to the scan entry .   
 */
void wlan_scan_entry_lock(wlan_scan_entry_t scan_entry);

/*
 *  
 * @release access to scan entry
 * ARGS :
 *  wlan_scan_entry_t        : handle to the scan entry .   
 */
void wlan_scan_entry_unlock(wlan_scan_entry_t scan_entry);

/*
 *  
 * @get mac address . 
 * ARGS :
 *  wlan_scan_entry_t        : handle to the scan entry .   
 * RETURNS:
 *  mac address of length IEEE80211_ADDR_LEN .
 */
u_int8_t * wlan_scan_entry_macaddr(wlan_scan_entry_t scan_entry);

/*
 *  
 * @get bssid . 
 * ARGS :
 *  wlan_scan_entry_t        : handle to the scan entry .   
 * RETURNS:
 *  bssid of length IEEE80211_ADDR_LEN .
 */
u_int8_t * wlan_scan_entry_bssid(wlan_scan_entry_t scan_entry);

/*
 *  
 * @get cap info . 
 * ARGS :
 *  wlan_scan_entry_t        : handle to the scan entry .   
 * RETURNS:
 *  cap info field .
 */
u_int16_t wlan_scan_entry_capinfo(wlan_scan_entry_t scan_entry);

/*
 *  
 * @get beacon interval . 
 * ARGS :
 *  wlan_scan_entry_t        : handle to the scan entry .   
 * RETURNS:
 *  beacon interval in TUs .
 */
u_int16_t wlan_scan_entry_beacon_interval(wlan_scan_entry_t scan_entry);

/*
 *  
 * @get beacon timestamp (tsf) .
 * ARGS :
 *  wlan_scan_entry_t        : handle to the scan entry .
 *  tsf                      : pointer to variable where timestamp will be copied.
 * RETURNS:
 *  beacon timestamp in TUs .
 */
void wlan_scan_entry_tsf(wlan_scan_entry_t scan_entry, u_int8_t *tsf);

/*
 *  
 * @reset "last received beacon" timestamp. 
 * ARGS :
 *  wlan_scan_entry_t        : handle to the scan entry .
 */
void wlan_scan_entry_reset_timestamp(wlan_scan_entry_t scan_entry);

/*
 *  
 * @get rssi . 
 * ARGS :
 *  wlan_scan_entry_t        : handle to the scan entry .   
 * RETURNS:
 *  average rssi of the scan entry (received beacon/probe response).
 */
u_int8_t wlan_scan_entry_rssi(wlan_scan_entry_t scan_entry);

/*
 * @get scan entry ssid . 
 * ARGS :
 *  wlan_scan_entry_t     : handle to the scan entry .   
 *  len                   : pointer to variable where the actual ssid length
 *                          will be copied
 * RETURNS:
 *      return a pointer to the ssid (NULL if length is 0)
 */
u_int8_t *wlan_scan_entry_ssid(wlan_scan_entry_t scan_entry, u_int8_t *len);

/*
 * @get DTIM period . 
 * ARGS :
 *  wlan_scan_entry_t     : handle to the scan entry .   
 * RETURNS:
 *      return the DTIM period
 */
u_int8_t wlan_scan_entry_dtimperiod(wlan_scan_entry_t scan_entry);

/*
 * @get TIM . 
 * ARGS :
 *  wlan_scan_entry_t     : handle to the scan entry .   
 * RETURNS:
 *      return TIM ie
 */
u_int8_t *wlan_scan_entry_tim(wlan_scan_entry_t scan_entry);

/*
 * @copy contents of beacon frame . 
 * ARGS :
 *  wlan_scan_entry_t        : handle to the scan entry .   
 *  beacon_buf               : buffer passed by the caller for the API to copy the beacon data.
 *                             if this buffer is not specified (set to NULL), the function
 *                             will simply return the total number of bytes used by the beacon.
 *  ie_len                   : caller passes the size of the beacon_buf and API implementation
 *                             returns the actual length of data copied.
 * RETURNS:
 *      return 0 on success.
 *      returns -ve value if length of the buffer passed in is insufficient. 
 */
int wlan_scan_entry_copy_beacon_data(wlan_scan_entry_t scan_entry, u_int8_t *beacon_buf, u_int16_t *beacon_len);

/*
 * @return beacon frame data . 
 * ARGS :
 *  wlan_scan_entry_t        : handle to the scan entry .   
 * RETURNS:
 *      pointer to beacon frame data.
 */
u_int8_t *wlan_scan_entry_beacon_data(wlan_scan_entry_t scan_entry);

/*
 * @get beacon frame length. 
 * ARGS :
 *  wlan_scan_entry_t        : handle to the scan entry .   
 * RETURNS:
 *      return length, in bytes, of the beacon frame.
 */
int wlan_scan_entry_beacon_len(wlan_scan_entry_t scan_entry);

/*
 * @copy ie information to the specified buffer. 
 * ARGS :
 *  wlan_scan_entry_t        : handle to the scan entry .   
 *  iebuf                    : buffer passed by the caller for the API to copy the IE data.
 *                             if this buffer is not specified (set to NULL), the function
 *                             will simply return the total number of bytes used by IE
 *                             elements.
 *  ie_len                   : caller passes the size of the iebuf and API implementation
 *                             returns the actual length of data copied.
 * RETURNS:
 *      return 0 on success.
 *      returns -ve value if length of the buffer passed in is insufficient. 
 */
int wlan_scan_entry_copy_ie_data(wlan_scan_entry_t scan_entry, u_int8_t *iebuf, u_int16_t *ie_len);

/*
 * @return ie blob. 
 * ARGS :
 *  wlan_scan_entry_t        : handle to the scan entry .   
 * RETURNS:
 *      pointer to the IE blob.
 */
u_int8_t *wlan_scan_entry_ie_data(wlan_scan_entry_t scan_entry);

/*
 * @get ie length. 
 * ARGS :
 *  wlan_scan_entry_t        : handle to the scan entry .   
 * RETURNS:
 *      return total byte count used by IEs.
 */
int wlan_scan_entry_ie_len(wlan_scan_entry_t scan_entry);

/*
 *  
 * @get channel. 
 * ARGS :
 *  wlan_scan_entry_t        : handle to the scan entry .   
 * RETURNS:
 *  channel of the scan entry .
 */
wlan_chan_t wlan_scan_entry_channel(wlan_scan_entry_t scan_entry);

/*
 *  
 * @get erp info. 
 * ARGS :
 *  wlan_scan_entry_t        : handle to the scan entry .   
 * RETURNS:
 *  erp info of the scan entry .
 */
u_int8_t wlan_scan_entry_erpinfo(wlan_scan_entry_t scan_entry);

/*
 * @get rates . 
 * ARGS :
 *  wlan_scan_entry_t        : handle to the scan entry .   
 */
u_int8_t *wlan_scan_entry_rates(wlan_scan_entry_t scan_entry);

/*
 * @get extended rates . 
 * ARGS :
 *  wlan_scan_entry_t        : handle to the scan entry .   
 */
u_int8_t *wlan_scan_entry_xrates(wlan_scan_entry_t scan_entry);

/*
 * @get RSN capability of a scan entry.
 * ARGS :
 *  wlan_scan_entry_t        : handle to the scan entry .   
 * RETURNS:
 *      pointer to rsn IE.
 */
int wlan_scan_entry_rsncaps(wlan_if_t vaphandle, wlan_scan_entry_t scan_entry, u_int16_t *rsncaps);

/*
 * @get SFA info . 
 * ARGS :
 *  wlan_scan_entry_t        : handle to the scan entry .   
 * RETURNS:
 *      pointer to SFA IE.
 */
u_int8_t *wlan_scan_entry_sfa(wlan_scan_entry_t scan_entry);

/*
 * @get Channel Switch Announcement (CSA) info . 
 * ARGS :
 *  wlan_scan_entry_t        : handle to the scan entry .   
 * RETURNS:
 *      pointer to CSA IE.
 */
u_int8_t *wlan_scan_entry_channelswitch(wlan_scan_entry_t scan_entry);

/*
 * @get Extended Channel Switch Announcement (XCSA) info . 
 * ARGS :
 *  wlan_scan_entry_t        : handle to the scan entry .   
 * RETURNS:
 *      pointer to XCSA IE.
 */
u_int8_t *wlan_scan_entry_extendedchannelswitch(wlan_scan_entry_t scan_entry);

/*
 * @get ht info . 
 * ARGS :
 *  wlan_scan_entry_t        : handle to the scan entry .   
 * RETURNS:
 *      return pointer to htinfo ie
 */
u_int8_t *wlan_scan_entry_htinfo(wlan_scan_entry_t scan_entry);

/*
 * @get ht info . 
 * ARGS :
 *  wlan_scan_entry_t        : handle to the scan entry .   
 * RETURNS:
 *      return pointer to htcap ie
 */
u_int8_t *wlan_scan_entry_htcap(wlan_scan_entry_t scan_entry);

/*
 * @get quiet info . 
 * ARGS :
 *  wlan_scan_entry_t        : handle to the scan entry .   
 * RETURNS:
 *      return pointer to quiet ie
 */
u_int8_t *wlan_scan_entry_quiet(wlan_scan_entry_t scan_entry);

/*
 * @get qbss load ie . 
 * ARGS :
 *  wlan_scan_entry_t        : handle to the scan entry .   
 * RETURNS:
 *      pointer to qbss load ie
 */
u_int8_t *wlan_scan_entry_qbssload(wlan_scan_entry_t scan_entry);

/*
 * @get Vendor-specific IE . 
 * ARGS :
 *  wlan_scan_entry_t        : handle to the scan entry .   
 * RETURNS:
 *      pointer to Vendor IE.
 */
u_int8_t *wlan_scan_entry_vendor(wlan_scan_entry_t scan_entry);

/*
 * @get ht info . 
 * ARGS :
 *  wlan_scan_entry_t        : handle to the scan entry .   
 *  country                  : char array of size 3 for API to return the 
 *                             null terminated 2 char country code. 
 * RETURNS:
 *      return 0 on success.
 *      returns -ve value if no country code is present. 
 */
int wlan_scan_entry_country(wlan_scan_entry_t scan_entry, u_int8_t *country);

/*
 * @get wme info . 
 * ARGS :
 *  wlan_scan_entry_t        : handle to the scan entry .   
 * RETURNS:
 *      pointer to WME Information IE.
 */
u_int8_t *wlan_scan_entry_wmeinfo_ie(wlan_scan_entry_t scan_entry);

/*
 * @get wme parameter. 
 * ARGS :
 *  wlan_scan_entry_t        : handle to the scan entry .   
 * RETURNS:
 *      pointer to WME Parameter IE.
 */
u_int8_t *wlan_scan_entry_wmeparam_ie(wlan_scan_entry_t scan_entry);

/*
 * @get age of the scan entry . 
 * ARGS :
 *  wlan_scan_entry_t        : handle to the scan entry .   
 * RETURNS:
 *  age of the scan entry in msec.
 */
u_int32_t wlan_scan_entry_age(wlan_scan_entry_t scan_entry);

/*
 * @get status of the scan entry . 
 * ARGS :
 *  wlan_scan_entry_t        : handle to the scan entry .   
 * RETURNS:
 *  status of the scan entry.
 *  intial value of the status to 0.
 */
u_int32_t wlan_scan_entry_status(wlan_scan_entry_t scan_entry);

/*
 * @get association state of the scan entry . 
 * ARGS :
 *  wlan_scan_entry_t        : handle to the scan entry .   
 * RETURNS:
 *  association state of the scan entry in msec.
 *  intial value of the status to 0.
 */
u_int32_t wlan_scan_entry_assoc_state(wlan_scan_entry_t scan_entry);

/*
 * @get scan entry bss type . 
 * ARGS :
 *  wlan_scan_entry_t     : handle to the scan entry .   
 * RETURNS:
 *      the bss type (infra or adhoc)
 */
enum ieee80211_opmode wlan_scan_entry_bss_type(wlan_scan_entry_t scan_entry);

/*
 * @get scan entry privacy . 
 * ARGS :
 *  wlan_scan_entry_t     : handle to the scan entry .   
 * RETURNS:
 *      the bss privacy bit (enabled/disabled)
 */
u_int8_t wlan_scan_entry_privacy(wlan_scan_entry_t scan_entry);

/*
 * @get Atheros capabilities . 
 * ARGS :
 *  wlan_scan_entry_t     : handle to the scan entry .   
 * RETURNS:
 *      pointer to Atheros capabilities ie
 */
u_int8_t *wlan_scan_entry_athcaps(wlan_scan_entry_t scan_entry);

/*
 * @get scan entry association cost . 
 * ARGS :
 *  wlan_scan_entry_t     : handle to the scan entry .   
 * RETURNS:
 *      the cost of associaton to this AP. This value is used when calculating
 *      the AP's rank.
 */
u_int32_t wlan_scan_entry_assoc_cost(wlan_scan_entry_t scan_entry);

/*
 * @get timestamp of the last failed association with the scan entry. 
 * ARGS :
 *  wlan_scan_entry_t     : handle to the scan entry .   
 * RETURNS:
 *      the time of the last failed association with the AP.
 */
systime_t wlan_scan_entry_bad_ap_timeout(wlan_scan_entry_t scan_entry);

/*
 * @get timestamp of scan entry's last association . 
 * ARGS :
 *  wlan_scan_entry_t     : handle to the scan entry .   
 * RETURNS:
 *  timestamp             : the time when the scan entry last associated to the home AP.
 */
systime_t wlan_scan_entry_lastassoc(wlan_scan_entry_t scan_entry);

/*
 * @get scan entry channel load
 * ARGS :
 *  wlan_scan_entry_t     : handle to the scan entry .   
 * RETURNS:
 *  utility               : The channel load of this particular entry (AP), 
 *                          used for load-balancing when selecting an AP for
 *                          association.
 */
u_int32_t wlan_scan_entry_chanload(wlan_scan_entry_t scan_entry);

/*
 * @get scan entry utility
 * ARGS :
 *  wlan_scan_entry_t     : handle to the scan entry .   
 * RETURNS:
 *  utility               : The utility of this particular entry, used to 
 *                          select the preferred AP for association or roaming.
 */
u_int32_t wlan_scan_entry_utility(wlan_scan_entry_t scan_entry);

/*
 * functions to set fields in scan entry 
 */
/*
 * @set status of the scan entry . 
 * ARGS :
 *  wlan_scan_entry_t     : handle to the scan entry .   
 *  status                : status word to be stored along with entry.
 *                          the value is not interpreted by the scanner and is opaque to the scanner . 
 *                          the caller needs store its own status word and needs to create its own 
 *                          interpretation for this status word.
 *                          a simple interpretation would  have 3 values  (0,1,2) for ststus word like
 *                          0: the association to this AP was succesful , 1: the association with AP failed once, 
 *                          2: the association failed with this AP multiple times.       
 * RETURNS:
 *      return 0 on success.
 *      returns -ve value on failure. 
 */
void wlan_scan_entry_set_status(wlan_scan_entry_t scan_entry, u_int32_t status);

/*
 * @set status of the scan entry . 
 * ARGS :
 *  wlan_scan_entry_t     : handle to the scan entry .   
 *  state                 : Association state.
 *                          the value is not interpreted by the scanner and is opaque to the scanner . 
 *                          the caller needs store its own association state and needs to create its own 
 *                          interpretation for this state.
 * RETURNS:
 *      return 0 on success.
 *      returns -ve value on failure. 
 */
void wlan_scan_entry_set_assoc_state(wlan_scan_entry_t scan_entry, u_int32_t state);

/*
 * @set association cost of the scan entry . 
 * ARGS :
 *  wlan_scan_entry_t     : handle to the scan entry .   
 *  cost                  : cost of association to this particular entry.
 *                          This is an arbitrary value set by the association 
 *                          module, to be used when calculating the AP's rank.
 */
void wlan_scan_entry_set_assoc_cost(wlan_scan_entry_t scan_entry, u_int32_t cost);

/*
 * @set timestamp of the last failed association with the scan entry.  
 * ARGS :
 *  wlan_scan_entry_t     : handle to the scan entry .   
 *  timestamp             : the time of the failed connection with the AP.
 */
void wlan_scan_entry_set_bad_ap_timeout(wlan_scan_entry_t scan_entry, systime_t timestamp);

/*
 * @set timestamp of scan entry's last association . 
 * ARGS :
 *  wlan_scan_entry_t     : handle to the scan entry .   
 *  timestamp             : the time when the scan entry last associated to the home AP.
 */
void wlan_scan_entry_set_lastassoc(wlan_scan_entry_t scan_entry, systime_t timestamp);

/*
 * functions to manipulate the scan entry's reference count
 */

/*
 * @get reference count of scan entry . 
 * ARGS :
 *  wlan_scan_entry_t     : handle to the scan entry .   
 * RETURNS:
 *      the reference count of scan entry. This value indicates how many
 *      clients are keeping a reference (pointer) to this entry.
 */
u_int8_t wlan_scan_entry_reference_count(wlan_scan_entry_t scan_entry);

/*
 * @add a reference count to an scan entry . 
 * ARGS :
 *  wlan_scan_entry_t     : handle to the scan entry .   
 * RETURNS:
 *      the new reference count of scan entry. This value indicates how many
 *      clients are keeping a reference (pointer) to this entry.
 */
u_int8_t wlan_scan_entry_add_reference(wlan_scan_entry_t scan_entry);

/*
 * @remove a reference count from an scan entry . 
 * ARGS :
 *  wlan_scan_entry_t     : handle to the scan entry .   
 * RETURNS:
 *      the new reference count of scan entry. This value indicates how many
 *      clients are keeping a reference (pointer) to this entry.
 */
u_int8_t wlan_scan_entry_remove_reference(wlan_scan_entry_t scan_entry);

/*
 * @print the names of scan notification events and reasons.
 * ARGS :
 *  event                 : event being notified .
 *  reason                : reason for the event .
 * RETURNS:
 *      a pointer to a constant string holding the event/reason description.
 */
const char* wlan_scan_notification_event_name(ieee80211_scan_event_type event);
const char* wlan_scan_notification_reason_name(ieee80211_scan_completion_reason reason);

/*
 * functions to fetch data from channel object 
 */

/*
 * @returns channel properties.
 * ARGS:
 *  chan                  : handle to channel object
 * Returns:
 *  wlan_channel_ieee     : the IEEE channel number
 *  wlan_channel_frequency: the fequency in Mhz
 *  wlan_channel_phymode  : the PHY mode
 *  wlan_channel_maxpower : maximum allowed transmit power for this channel
 *  wlan_channel_flags    : propitiery channel flags
 *  wlan_channel_is_passive :returns true if passive channel 
 */
u_int32_t wlan_channel_ieee(wlan_chan_t chan);
u_int32_t wlan_channel_frequency(wlan_chan_t chan);
enum ieee80211_phymode wlan_channel_phymode(wlan_chan_t chan);
int8_t wlan_channel_maxpower(wlan_chan_t chan);
u_int32_t wlan_channel_flags(wlan_chan_t chan);
bool wlan_channel_is_passive(wlan_chan_t chan);


/*
 * helper functions to convert between frequency and channel number.
 */
u_int32_t wlan_mhz2ieee(wlan_dev_t devhandle, u_int32_t  frequency);
u_int32_t wlan_ieee2mhz(wlan_dev_t devhandle, int ieeechan);

/* public methods of the VAP object */ 

/*
 * @sets channel.
 * ARGS: 
 *  wlan_if_t             : handle to the vap.
 *  int chan              : ieee channel #.
 * RETURNS:
 *  on success returns 0. 
 *  on failure returns a negative value.
 *  on succesfull channel change it posts a channel change event. 
 *
 * if the vap is running and if the channel is IEEE80211_CHAN_ANY then 
 *    the implementation chooses any appropriate channel
 *    when one of the vaps is brought up. In this case no channel cange event is posted.  
 * if the vap is up and running and if the channel is IEEE80211_CHAN_ANY  
 *    then the implementation would return ana error.
 * if the vap is running and if the channel is a valid channel then 
 *    the implementation would set the channel to be the desired channel for the current phymode.
 *    no channel change event will be posted in this case..
 */
int wlan_set_channel(wlan_if_t vaphandle, int chan);

/*
 * @returns current channel.
 * ARGS: 
 *  ieee80211_if_t        : handle to the vap .   
 * RETURNS:
 *  on success returns current channel.
 *  if no current channel is set it returns 0.
 */
wlan_chan_t wlan_get_current_channel(wlan_if_t vaphandle);

/*
 * @returns current channel.
 * ARGS: 
 *  ieee80211_if_t        : handle to the vap .   
 *  int chan              : ieee channel  #.
 * RETURNS:
 *  on success returns bss ieee80211_channel. 
 *  returns 0 if no bss channel is present( i.e we are not associated). 
 */
wlan_chan_t wlan_get_bss_channel(wlan_if_t vaphandle);

/*
 * @returns current list of channels of a certain wireless mode.
 * ARGS: 
 *  wlan_dev_t            : handle to the device.   
 *  chanlist              : array of channel list to be filled in.
 *  bsschan_idx           : index of bss channel in the array.
 *  nchan                 : length of the chanlist array passed in.
 * RETURNS:
 *  on success returns number of channels filled into the chanlist array. 
 *  returns  a -ve value of failure. if nchan is 0 cna chanlist is null 
 * then it returns the number of channels. 
 */
int wlan_get_channel_list(wlan_dev_t devhandle, enum ieee80211_phymode phymode,
                          wlan_chan_t chanlist[], int *bsschan_idx, u_int32_t nchan);

/*
 * @returns set of supported phy modes.
 *   ARGS:
 *       devhandle   : device handle
 *       modes       : array of supported modes (the caller owns the memory).
 *       nmodes      : number of supported modes
 *       len         : input array len
 *   RETURNS:
 *       on success it returns the number of phy modes filled in the modes array..
 *       return -ve value on failure.
 */
int wlan_get_supported_phymodes(wlan_dev_t devhandle,
                                enum ieee80211_phymode *modes,
                                u_int16_t *nmodes,
                                u_int16_t len);

/*
 * @returns the desired phy mode of current interface.
 *   ARGS:
 *       vaphandle   : vap handle
 *   RETURNS:
 *       on success it returns the desired phy mode
 */
enum ieee80211_phymode wlan_get_desired_phymode(wlan_if_t vaphandle);

/*
 * @set the desired phy mode of current interface.
 *   ARGS:
 *       vaphandle   : vap handle
 *       mode        : the mode to set
 */
int wlan_set_desired_phymode(wlan_if_t vaphandle, enum ieee80211_phymode mode);

/*
 * @returns the current phy mode of current interface.
 *   ARGS:
 *       vaphandle   : vap handle
 *   RETURNS:
 *       on success it returns the current phy mode
 */
enum ieee80211_phymode wlan_get_current_phymode(wlan_if_t vaphandle);

/*
 * @returns the BSS phy mode of current associated AP.
 *   ARGS:
 *       vaphandle   : vap handle
 *   RETURNS:
 *       on success it returns the BSS phy mode
 */
enum ieee80211_phymode wlan_get_bss_phymode(wlan_if_t vaphandle);

/*
 * @sets country code. should allow when the interface is up ?
 * ARGS: 
 *  ieee80211_dev_t    : handle to the radio .   
 *  char *cc           : null terminated 2 char country code string.
 *                        
 * RETURNS:
 *  on success returns 0. 
 *  on failure returns a negative value.
 *  on succesfull country code change it posts a IEEE80211_CC_CHANGED event. 
 */
int wlan_set_countrycode(wlan_dev_t devhandle, char cc[3]);

/*
 * @returns current country code.
 * ARGS: 
 *  ieee80211_dev_t    : handle to the radio .   
 * RETURNS:
 *  on success returns current country code. 
 *  if no country code is set it returns NULL.
 *  the returned country code string points to internal memory, if the
 *  caller needs to make a copy if it needs to cache it.
 */
int wlan_get_countrycode(wlan_dev_t devhandle, char cc[3]);

/*
 * @returns current regulatory domain.
 * ARGS: 
 *  devhandle          : handle to the radio.
 * RETURNS:
 *  returns current regdomain code. 
 */
u_int16_t wlan_get_regdomain(wlan_dev_t devhandle);

/*
 * @set regulatory domain.
 * ARGS: 
 *  devhandle          : handle to the radio.
 *  regdmn             : regdomain code.
 */
int wlan_set_regdomain(wlan_dev_t devhandle, u_int16_t regdmn);

/*
 * @gets the interface address (mac address) 
 *  ARGS:
 *       vaphandle : vap handle
 *  RETURNS:
 *        interface MAC address array of length IEEE80211_ADDR_LEN 
 */
u_int8_t *wlan_vap_get_macaddr(wlan_if_t vaphandle);

/*
 * @gets the hw interface address (mac address defined in EEPROM) 
 *  ARGS:
 *       vaphandle : vap handle
 *  RETURNS:
 *        interface MAC address array of length IEEE80211_ADDR_LEN 
 */
u_int8_t *wlan_vap_get_hw_macaddr(wlan_if_t vaphandle);

/* @get the BSSID(mac address) 
 *  ARGS:
 *   wlan_if_t  : handle to the vap .   
 *  RETURNS:
 *        address array of length IEEE80211_ADDR_LEN 
 */
int wlan_vap_get_bssid(wlan_if_t vaphandle, u_int8_t *bssid);

/* @get the common device handle 
 *  ARGS:
 *   wlan_if_t  : handle to the vap .   
 *  RETURNS:
 *        dev_handle. 
 */
wlan_dev_t wlan_vap_get_devhandle(wlan_if_t vaphandle);


/*
 * @returns the supproted rates for a certain phymode
 *  ARGS:
 *       devhandle: device handle
 *       phymode  : the phymode
 *       rates    : the rate array
 *       len      : array len
 *       nrates   : number of supported rates
 *  RETURNS:
 *       On success the rates array will be filled and nrates set to the number
 *       of supported rates, or -ve otherwise.
 */
int wlan_get_supported_rates(wlan_dev_t devhandle, enum ieee80211_phymode mode,
                             u_int8_t *rates, u_int32_t len, u_int32_t *nrates);

/*
 * @returns the current operational rates for a certain phymode
 *  ARGS:
 *       vaphandle: vap handle
 *       phymode  : the phymode
 *       rates    : the rate array
 *       len      : array len
 *       nrates   : number of operational rates
 *  RETURNS:
 *       On success the rates array will be filled and nrates set to the number
 *       of operational rates, or -ve otherwise.
 */
int wlan_get_operational_rates(wlan_if_t vaphandle, enum ieee80211_phymode mode,
                               u_int8_t *rates, u_int32_t len, u_int32_t *nrates);

/*
 * @set the current operational rates for a certain phymode
 *  ARGS:
 *       vaphandle: vap handle
 *       phymode  : the phymode
 *       rates    : the rate array
 *       nrates   : number of operational rates
 */
int wlan_set_operational_rates(wlan_if_t vaphandle, enum ieee80211_phymode mode,
                               u_int8_t *rates, u_int32_t nrates);

/**
 * @query the operational mode of a vap.
 * ARGS :
 *  wlan_if_t     : vap handle.   
 *
 * RETURNS:
 *  the current operational mode of vap.
 */
enum ieee80211_opmode wlan_vap_get_opmode(wlan_if_t vaphandle);

/*
 * @get rssi info 
 * ARGS :
 * wlan_if_t             : vap handle.
 *  rssitype             : type of rssi. 
 *  wlan_rssi_info       : handle for the return value .   
 *
 *  RETURNS: 0  on success and -ve on failure.
 *           only valid for STA vaps;
 *
 */
int wlan_getrssi(wlan_if_t vaphandle, wlan_rssi_info *rssi_info, wlan_rssi_type rssitype);


/* 
 * @get tx rate info for the bss.
 * ARGS :
 *  wlan_node_t          : handle to a vap .   
 *  rate_info            : tx rate info for the bss.
 *  RETURNS: 0  on success and -ve on failure.
 *           only valid for STA vaps;
 */
int wlan_get_txrate_info(wlan_if_t vaphandle, ieee80211_rate_info *rate_info);

/*
 * @set the current bss rates 
 *  ARGS:
 *       vaphandle: vap handle
 *       rates    : the rate array
 *       len      : length of the array.
 *       nrates   : number of operational rates.
 */
int wlan_get_bss_rates(wlan_if_t vaphandle, u_int8_t *rates, 
                           u_int32_t len, u_int32_t *nrates);

/*
 * @set the current bss ht rates 
 *  ARGS:
 *       vaphandle: vap handle
 *       rates    : the rate array
 *       len      : length of the array.
 *       nrates   : number of operational rates.
 */
int wlan_get_bss_ht_rates(wlan_if_t vaphandle, u_int8_t *rates, 
                           u_int32_t len, u_int32_t *nrates);

/*
 * public methods to node object 
 */

/*
 * @get rssi info 
 * ARGS :
 * wlan_node_t           : node handle.
 *  rssitype             : type of rssi. 
 *  wlan_rssi_info       : handle for the return value .   
 *
 *  RETURNS: 0  on success and -ve on failure.
 *
 */
int wlan_node_getrssi(wlan_node_t node, wlan_rssi_info *rssi_info, wlan_rssi_type rssitype);

/*
 * @get mac address of a node/station
 * ARGS :
 *  wlan_node_t          : handle to a node/station .   
 *
 *  RETURNS: mac address on success  and NULL on failure.
 *
 */
u_int8_t *wlan_node_getmacaddr(wlan_node_t node);

/*
 * @get BSSID of BSS that node/station is connected to.
 * ARGS :
 *  wlan_node_t          : handle to a node/station .   
 *
 *  RETURNS: BSSID.
 *
 */
u_int8_t *wlan_node_getbssid(wlan_node_t node);

/*
 * @get capinfo of a node/station
 * ARGS :
 *  wlan_node_t          : handle to a node/station .   
 *
 *  RETURNS: capinfo.
 *
 */
u_int16_t wlan_node_getcapinfo(wlan_node_t node);

/* 
 * @get tx rate info for the node.
 * ARGS :
 *  wlan_node_t          : handle to a node/station .   
 * RETURNS: tx rate info for the node.
 */
int wlan_node_txrate_info(wlan_node_t node, ieee80211_rate_info *rate_info);

/* 
 * @get wpa ie from a node.
 * ARGS :
 *       wlan_if_t          : handle to a vap
 *       macaddr: mac address of the node.
 *       ie     : ie to return.
 *       len    : length of the array paaed in.
 *                len is in/out param.
 *               
 * RETURNS: 
 *       0 on success and -ve value on failure.
 *       on success the actual length of data copied into
 *       ie structure is returned in len.
 */
int  wlan_node_getwpaie(wlan_if_t vap, u_int8_t *macaddr, u_int8_t *ie, u_int16_t *len);

/* 
 * @authorize/unauthorize a node.
 * ARGS:
 *  wlan_if_t          : handle to a vap
 *  open               : 1 to authorize, 0 to unauthorize
 *  macaddr            : peer MAC address of the node
 */
int wlan_node_authorize(wlan_if_t vaphandle, int authorize, u_int8_t *macaddr);

u_int32_t wlan_node_param(wlan_node_t node, ieee80211_node_param_type param);

/* 
 * @handle association decision from OS
 * ARGS:
 *  wlan_if_t          : handle to a vap
 *  macaddr            : peer MAC address of the node
 *  assoc_status       : association status code to be set
 * RETURNS: 
 *       0 on success and -ve value on failure.
 */
u_int32_t wlan_node_set_assoc_decision(wlan_if_t vap, u_int8_t *macaddr, u_int16_t assoc_status);

/* 
 * @retrieve cached OS association decision
 * ARGS:
 *  wlan_if_t          : handle to a vap
 *  macaddr            : peer MAC address of the node
 * RETURNS: 
 *       Cached association decision.
 */
u_int32_t wlan_node_get_assoc_decision(wlan_if_t vap, u_int8_t *macaddr);

/*
 * @get statistics about a node/station
 * ARGS :
 *  wlan_node_t          : handle to a node/station .   
 *
 *  RETURNS: node statistics  on success  and NULL on failure.
 *
 */

struct ieee80211_nodestats *wlan_node_stats(wlan_node_t node);


typedef void (* ieee80211_sta_iter_func) (void *, wlan_node_t);
/*
 *  
 * @returns info about list of current of stations.
 * ARGS: 
 *  wlan_if_t       : handle to the vap .   
 *  iter_func       : function called for each station in the table..
 *  arg             : arg passed bak to the iter function.
 * RETURNS:
 * it returns the number of stas(nodes). if the iter_func is not null
 * the iter_func is called for each node in the context of the caller thread. 
 * for AP mode VAP this API iters through the list of associated stations.
 * for STA mode VAP this API iters through the one node corresponding to AP.
 * for IBSS mode VAP this API iters through the list of neighbors in the IBSS network.
 */
int32_t wlan_iterate_station_list(wlan_if_t vaphandle,ieee80211_sta_iter_func iter_func,void *arg);


/*
 * @set auth mode.
 * ARGS :
 *  ieee80211_if_t       : handle to the vap.
 *  modes                : auth mode array.
 *  len                  : number of auth modes
 *  RETURNS: 0  on success and -ve on failure.
 */
int wlan_set_authmodes(wlan_if_t vaphandle, ieee80211_auth_mode modes[], u_int len);

/*
 * @get auth mode.
 * ARGS :
 *  wlan_if_t            : handle to the vap .   
 *  types                :  array for the supported cipher types to be filled in .
 *  len                  :  length of the passed in types array .
 *  RETURNS: on success number of elements filled in to modes array and -ve on failure.
 */
u_int wlan_get_auth_modes(wlan_if_t vaphandle, ieee80211_auth_mode modes[], u_int len);

/*
 * @set unicast cipher set.  
 * 
 * ARGS :
 *  wlan_if_t            : handle to the vap   
 *  ctypes               : array of cipher types
 *  len                  : number of cipher types
 *  RETURNS: 0  on success and -ve on failure.
 */
int wlan_set_ucast_ciphers(wlan_if_t vaphandle, ieee80211_cipher_type types[], u_int len);

/*
 * @set multicast cipher set.
 * 
 * ARGS :
 *  wlan_if_t            : handle to the vap   
 *  ctypes               : array of cipher types
 *  len                  : number of cipher types
 *  RETURNS: 0  on success and -ve on failure.
 */
int wlan_set_mcast_ciphers(wlan_if_t vaphandle, ieee80211_cipher_type types[], u_int len);

/*
 * @get unicast cipher types.  
 * 
 * ARGS :
 *  wlan_if_t            :  handle to the vap.
 *  types                :  array for the supported cipher types to be filled in.
 *  len                  :  length of the passed in types array .
 *  RETURNS: on success number of elements filled in to types array and -ve on failure.
 *           if len is 0 and types is NULL then it would return the  number of types
 *           stored in the rsn IE internally.
 */
u_int wlan_get_ucast_ciphers(wlan_if_t vaphandle, ieee80211_cipher_type types[], u_int len);

/*
 * @get multicast cipher types.  
 * 
 * ARGS :
 *  wlan_if_t            :  handle to the vap.
 *  types                :  array for the supported cipher types to be filled in.
 *  len                  :  length of the passed in types array .
 *  RETURNS: on success number of elements filled in to types array and -ve on failure.
 *           if len is 0 and types is NULL then it would return the  number of types
 *           stored in the rsn IE internally.
 */
u_int wlan_get_mcast_ciphers(wlan_if_t vaphandle, ieee80211_cipher_type types[], u_int len);

/*
 * @get active multicast cipher type (STA only).
 * 
 * ARGS :
 *  wlan_if_t            :  handle to the vap.
 *  RETURNS: the active (current) multicast cipher.
 */
ieee80211_cipher_type wlan_get_current_mcast_cipher(wlan_if_t vaphandle);

/*
 * @set cipher params in rsn IE.
 * 
 * ARGS :
 *  wlan_if_t            : handle to the vap.   
 *  type                 : parameter type.
 *  val                  : value of the parameter. 
 *  RETURNS: 0  on success and -ve on failure.
 */
int wlan_set_rsn_cipher_param(wlan_if_t vaphandle, ieee80211_rsn_param type, u_int32_t val);

/*
 * @get cipher params in rsn IE.  
 * 
 * ARGS :
 *  wlan_if_t            : handle to the vap.   
 *  type                 : parameter type.
 *  RETURNS: parameter  on success and -ve on failure.
 */
int32_t wlan_get_rsn_cipher_param(wlan_if_t vaphandle, ieee80211_rsn_param type);

/*
 * @set default WEP Key.
 * ARGS :
 *  wlan_if_t            : handle to the vap .   
 *  keyindex             : key index. IEEE80211_KEYIX_NONE for keymapping keys, otherwise default keys.
 *  key                  : key value
 *  RETURNS: 0  on success and -ve on failure.
 */
int wlan_set_key(wlan_if_t vaphandle, u_int16_t keyix, ieee80211_keyval *key);

/*
 * @set default WEP Key ID.
 * ARGS :
 *  wlan_if_t            : handle to the vap .   
 *  keyindex             : key index .
 *  RETURNS: 0  on success and -ve on failure.
 */
int wlan_set_default_keyid(wlan_if_t vaphandle, u_int keyix);

/*
 * @set default tx Wep Key.
 * ARGS :
 *  wlan_if_t       : handle to the vap .   
 *  RETURNS: returns key index on success and -ve on failure.
 */
u_int16_t wlan_get_default_keyid(wlan_if_t vaphandle);

/*
 * @get encryption key.
 * 
 * ARGS :
 *  wlan_if_t            : handle to the vap.   
 *  RETURNS: wpa key.
 */
int wlan_get_key(wlan_if_t vaphandle, u_int16_t keyix, u_int8_t *macaddr, ieee80211_keyval *key, u_int16_t keybuf_len);

/*
 * @delete encryption key.
 * 
 * ARGS :
 *  wlan_if_t            : handle to the vap.
 *  keyid                : key id.
 *  macaddr              : mac address of the node to delete the key for. 
 *                         all bytes are 0xff for  multicast key. 
 *  RETURNS: 0  on success and -ve on failure.
 */
int wlan_del_key(wlan_if_t vaphandle, u_int16_t keyid, u_int8_t *macaddr);

/*
 * @restore encryption keys.
 * 
 * ARGS :
 *  wlan_if_t            : handle to the vap.
 */
void wlan_restore_keys(wlan_if_t vaphandle);

/*
 * @set Powersave mode .
 * 
 * ARGS :
 *  wlan_if_t       : handle to the vap .   
 *  mode            : power save mode .
 *  RETURNS: 0  on success and -ve on failure.
 */
int wlan_set_powersave(wlan_if_t vaphandle, ieee80211_pwrsave_mode mode);

/*
 * @get Powersave mode .
 * 
 * ARGS :
 *  wlan_if_t       : handle to the vap .   
 *  RETURNS: power save mode.
 */
ieee80211_pwrsave_mode wlan_get_powersave(wlan_if_t vaphandle);

/*
 * @get Atheros application specific powersave state.
 * 
 * ARGS :
 *  wlan_if_t       : handle to the vap .   
 *  RETURNS: power save state.
 */
u_int32_t wlan_get_apps_powersave_state(wlan_if_t vaphandle);

/*
 * @set MIMO powersave mode .
 * 
 * ARGS :
 *  wlan_if_t       : handle to the vap .   
 *  mode            : mimo power save mode .
 *  RETURNS: 0  on success and -ve on failure.
 */
int wlan_set_mimo_powersave(wlan_if_t vaphandle, ieee80211_mimo_powersave_mode mode);

/*
 * @get mimo Powersave mode .
 * 
 * ARGS :
 *  wlan_if_t       : handle to the vap .   
 *  RETURNS: mimo power save mode.
 */
ieee80211_mimo_powersave_mode wlan_get_mimo_powersave(wlan_if_t vaphandle);



/*
 * @set simple configuration parameter. 
 * 
 * ARGS :
 *  wlan_if_t       : handle to the vap.
 *  param           : simple config paramaeter.
 *  val             : value of the parameter.
 *  RETURNS: 0  on success and -ve on failure.
 */
int wlan_set_param(wlan_if_t vaphandle, ieee80211_param param, u_int32_t val);

/*
 * @get simple configuration parameter. 
 * 
 * ARGS :
 *  wlan_if_t       : handle to the vap.
 *  param           : simple config paramaeter.
 *  RETURNS:value of the parameter. 
 */
u_int32_t wlan_get_param(wlan_if_t vaphandle, ieee80211_param param);

/*
 * @set fixed rate. 
 * 
 * ARGS :
 *  wlan_if_t       : handle to the vap (AP vap) .   
 *  rate            : simple config paramaeter.
 *  RETURNS: 0  on success and -ve on failure.
 *          passing a NULL value for the rate clears fixed rate setting.
 */
int wlan_set_fixed_rate(wlan_if_t vaphandle, ieee80211_rate_info *rate);


/*
 * @get fixed rate. 
 * 
 * ARGS :
 *  wlan_if_t       : handle to the vap (AP vap) .   
 *  rate_info       : fixed rate info. 
 *  RETURNS: 0  on success and -ve on failure.
 *           returns  ENOENT if no fixed rate.
 */
int wlan_get_fixed_rate(wlan_if_t vaphandle, ieee80211_rate_info *rate);


/*
 * @set ACL parameters.
 * 
 * ARGS :
 *  wlan_if_t       : handle to the vap (AP vap) .   
 *  aclcmd          :  array of supported key types .
 *  macaddr         :  mac address. only valid for ADDMAC,DELMAC acl commands,
 *  RETURNS: 0  on success and -ve on failure.
 */
int wlan_set_acl_param(wlan_if_t vaphandle,ieee80211_acl_cmd aclcmd,u_int8_t *macaddr);

    
/*
 * @set wmm parameter.
 * 
 * ARGS :
 *  wlan_if_t            : handle to the vap (AP vap) .   
 *  wmeparam             :  wme parameter .
 *  isbss                :  1: bss param 0: local param
 *  ac                   : access category.   
 *   val                 :  value of the parameter.
 *  RETURNS: 0  on success and -ve on failure.
 */

int wlan_set_wmm_param(wlan_if_t vaphandle,wlan_wme_param wmeparam,
                            u_int8_t isbss, u_int8_t ac,u_int32_t val);
 
/*
 * @get wmm parameter.
 * 
 * ARGS :
 *  wlan_if_t       : handle to the vap (AP vap) .   
 *  wmeparam        :  array of supported key types .
 *  isbss           :  1: bss param 0: local param
 *  ac              : access category.   
 *  RETURNS: value of the parameter.
 */

u_int32_t wlan_get_wmm_param(wlan_if_t vaphandle,wlan_wme_param wmeparam,
                            u_int8_t isbss, u_int8_t ac);

/*
 * @set ampdu mode..
 * 
 * ARGS :
 *  wlan_if_t       : handle to the vap (AP vap) .   
 *  mode            : ampdu mode.
 *  RETURNS: 0  on success and -ve on failure.
 */
int wlan_set_ampdu_mode(wlan_if_t vaphandle, ieee80211_ampdu_mode mode);


/*
 * @get ampdu mode..
 * 
 * ARGS :
 *  wlan_if_t       : handle to the vap (AP vap) .   
 *  RETURNS: ampdu mode.
 */
ieee80211_ampdu_mode wlan_get_ampdu_mode(wlan_if_t vaphandle);

/*
 * @get vap ststistics..
 * 
 * ARGS :
 *  wlan_if_t       : handle to the vap (AP vap) .   
 *  RETURNS: vap stats 
 */
struct ieee80211_stats *wlan_get_stats(wlan_if_t vaphandle);


/*
 * @send frame.
 *
 *  wlan_if_t       : handle to the vap (AP vap) .   
 *  macaddr         : destination mac address.
 *  frame           :  type of frame to send.
 *  RETURNS: 0  on success and -ve on failure.
 */
int wlan_send_frame(wlan_if_t vaphandle,ieee80211_send_frame_type frame);

/*
 * @set up privacy filters.
 *
 *  wlan_if_t       : handle to the vap.
 *  filters         : array of privacy filters.
 *  num_filters     : length of the filters array.
 *  RETURNS: 0  on success and -ve on failure.
 *           
 */
int wlan_set_privacy_filters(wlan_if_t vaphandle, ieee80211_privacy_exemption *filters, u_int32_t num_filters);

/*
 * @query privacy filters.
 *
 *  wlan_if_t       : handle to the vap.
 *  filters         : array of privacy filters.
 *  num_filters     : number of the filters.
 *  len             : size of the filters array.
 *  RETURNS: on success number of elements filled in to filters array and -ve on failure.
 */
int wlan_get_privacy_filters(wlan_if_t vaphandle, ieee80211_privacy_exemption *filters, u_int32_t *num_filters, u_int32_t len);

/*
 * @set up PMKID list.
 *
 *  wlan_if_t       : handle to the vap.
 *  pmkids          : array of PMKID entries.
 *  num             : length of the pmkids array.
 *  RETURNS: 0  on success and -ve on failure.
 *           
 */
int wlan_set_pmkid_list(wlan_if_t vaphandle, ieee80211_pmkid_entry *pmkids, u_int16_t num);

/*
 * @query PMKID list.
 *
 *  wlan_if_t       : handle to the vap.
 *  pmkids          : array of PMKID entries.
 *  count           : number of the PMKID entries.
 *  len             : size of the pmkids array.
 *  RETURNS: on success number of elements filled in to pmkids array and -ve on failure.
 */
int wlan_get_pmkid_list(wlan_if_t vaphandle, ieee80211_pmkid_entry *pmkids, u_int16_t *count, u_int16_t len);

/*
 * MLME public methods
 */

/*
 * @MLME Join Request (Infrastructure)
 *
 * wlan_if_t            : handle to the station vap
 * bss_entry            : scan entry of the desired BSS
 * timeout              : time out value for join request (number of beacon intervals)
 * RETURNS:             : IEEE80211 status code
 */
int wlan_mlme_join_infra(wlan_if_t vaphandle, wlan_scan_entry_t bss_entry, u_int32_t timeout);

/*
 * @MLME Auth Request
 *
 * wlan_if_t            : handle to the station vap
 * timeout              : time out value for auth request (in ms)
 * RETURNS:             : IEEE80211 status code
 */
int wlan_mlme_auth_request(wlan_if_t vaphandle, u_int32_t timeout);

/*
 * @MLME Deauth Request
 *
 * wlan_if_t            : handle to the vap
 * macaddr              : peer MAC address
 * reason               : IEEE80211 reason code
 * RETURNS:             : IEEE80211 status code
 */
int wlan_mlme_deauth_request(wlan_if_t vaphandle, u_int8_t *macaddr, IEEE80211_REASON_CODE reason);

/*
 * @MLME Association Request
 *
 * wlan_if_t            : handle to the station vap
 * timeout              : time out value for assoc request (ms)
 * RETURNS:             : IEEE80211 status code
 */
int wlan_mlme_assoc_request(wlan_if_t vaphandle, u_int32_t timeout);

/*
 * @MLME Reassociation Request
 *
 * wlan_if_t            : handle to the station vap
 * prev_bssid           : previous BSSID
 * timeout              : time out value for assoc request (ms)
 * RETURNS:             : IEEE80211 status code
 */
int wlan_mlme_reassoc_request(wlan_if_t vaphandle, u_int8_t *prev_bssid, u_int32_t timeout);

/*
 * @MLME Disassoc Request
 *
 * wlan_if_t            : handle to the vap
 * macaddr              : peer MAC address
 * reason               : IEEE80211 reason code
 * RETURNS:             : IEEE80211 status code
 */
int wlan_mlme_disassoc_request(wlan_if_t vaphandle, u_int8_t *macaddr, IEEE80211_REASON_CODE reason);

/*
 * @MLME Join Request (IBSS)
 *
 * wlan_if_t            : handle to the station vap
 * bss_entry            : scan entry of the desired BSS
 * timeout              : time out value for join request
 * RETURNS:             : IEEE80211 status code
 */
int wlan_mlme_join_adhoc(wlan_if_t vaphandle, wlan_node_t node, u_int32_t timeout);

/*
 * @MLME Start BSS (AP/IBSS)
 *
 * wlan_if_t            : handle to the vap
 * RETURNS:             : IEEE80211 status code
 */
int wlan_mlme_start_bss(wlan_if_t vaphandle);

/*
 * @MLME Stop BSS (AP/IBSS)
 *
 * wlan_if_t            : handle to the vap
 * RETURNS:             : IEEE80211 status code
 */
int wlan_mlme_stop_bss(wlan_if_t vaphandle, int flags);
#define WLAN_MLME_STOP_BSS_F_SEND_DEAUTH                0x1
#define WLAN_MLME_STOP_BSS_F_FLUSH_NODES                0x2
#define WLAN_MLME_STOP_BSS_F_CLEAR_ASSOC_STATE          0x4

/*
 * @MLME Pause BSS (AP/IBSS)
 *
 * wlan_if_t            : handle to the station vap
 * RETURNS:             : IEEE80211 status code
 */
int wlan_mlme_pause_bss(wlan_if_t vaphandle);

/*
 * @MLME Resume BSS (AP/IBSS)
 *
 * wlan_if_t            : handle to the vap
 * RETURNS:             : IEEE80211 status code
 */
int wlan_mlme_resume_bss(wlan_if_t vaphandle);

/*
 * @MLME Cancel
 *
 * wlan_if_t            : handle to the station vap
 * RETURNS:             : IEEE80211 status code
 */
int wlan_mlme_cancel(wlan_if_t vaphandle);

/*
 * @MLME Start monitoring (monitor mode)
 *
 * wlan_if_t            : handle to the station vap
 * RETURNS:             : IEEE80211 status code
 */
int wlan_mlme_start_monitor(wlan_if_t vaphandle);

/*
 * @set application specific ie buffer for a given frame..
 * 
 * ARGS :
 *  wlan_if_t       : handle to the vap (AP vap) .   
 *  frame type      : frame type.   
 *  buf             : appie buffer.
 *  buflen          : length of the buffer.
 *  RETURNS: 0  on success and -ve on failure.
 * once the buffer is set for a perticular frame the MLME keep setting up the ie in to the frame 
 * every time the frame is sent out. passing 0 value to buflen would remove the appie biffer.
 */
int wlan_mlme_set_appie(wlan_if_t vaphandle, ieee80211_frame_type ftype, u_int8_t *buf, u_int16_t buflen);

/*
 * @get application specific ie buffer for a given frame..
 * 
 * ARGS :
 *  wlan_if_t       : handle to the vap (AP vap) .   
 *  frame type      : frame type.   
 *  buf             : appie buffer.
 *  buflen          : length of the buffer.
 *  RETURNS: 0 on success and buf is filled in with specific IE and -ve on failure.
 */
int wlan_mlme_get_appie(wlan_if_t vaphandle, ieee80211_frame_type ftype, u_int8_t *buf, u_int32_t *ielen, u_int32_t buflen);

/*
 * @MLME Get Link Rate
 *
 * wlan_if_t            : handle to the station vap
 * RETURNS:             : link rate (bps)
 */
u_int32_t wlan_get_linkrate(wlan_if_t vaphandle);


/*
 * @MLME Get rssi (adhoc)
 *
 * wlan_if_t            : handle to the station vap
 * RETURNS:             : RSSI
 */
int32_t wlan_get_rssi_adhoc(wlan_if_t vaphandle);

/*
 * @MLME Set connection state "up"
 *
 * wlan_if_t            : handle to the vap
 * RETURNS:             : 
 */
void wlan_mlme_connection_up(wlan_if_t vaphandle);

/*
 * @MLME Set connection state "down"
 *
 * wlan_if_t            : handle to the vap
 * RETURNS:             : 
 */
void wlan_mlme_connection_down(wlan_if_t vaphandle);

/*
 * @MLME Reset Connection
 *
 * wlan_if_t            : handle to the vap
 * RETURNS:             : IEEE80211 status code
 */
int wlan_mlme_connection_reset(wlan_if_t vaphandle);


/*
 * @MLME Get timeout in ms for disassociating nodes
 *
 * wlan_if_t            : handle to the station vap
 * RETURNS:             : timeout (ms)
 */
u_int32_t wlan_get_disassoc_timeout(wlan_if_t vaphandle);

/*
 * @MLME Set timeout in ms for disassociating nodes
 *
 * wlan_if_t            : handle to the station vap
 * node_age_timeout     : timeout (ms)
 * RETURNS:             : 
 */
void wlan_set_disassoc_timeout(wlan_if_t vaphandle, u_int32_t node_age_timeout);

/*
 * @MLME Build candidate list of nodes (adhoc) 
 *
 * wlan_if_t            : handle to the station vap
 * RETURNS:             : 
 */
void wlan_mlme_adhoc_candidate_list_build(wlan_if_t vaphandle);

/*
 * @MLME Free candidate list (adhoc)
 *
 * wlan_if_t            : handle to the station vap
 * RETURNS:             : 
 */
void wlan_mlme_adhoc_candidate_list_free(wlan_if_t vaphandle);

/*
 * @MLME Number of nodes in candidate list (adhoc)
 *
 * wlan_if_t            : handle to the station vap
 * RETURNS:             : 
 */
u_int32_t wlan_mlme_adhoc_candidate_list_count(wlan_if_t vaphandle);

/*
 * @MLME Return node at given index in candidate list (adhoc) 
 *
 * wlan_if_t            : handle to the station vap
 * u_int32_t            : index 
 * RETURNS:             : wlan_node_t
 */
wlan_node_t wlan_mlme_adhoc_candidate_list_get(wlan_if_t vaphandle, u_int32_t index);

/*
 * @MLME Build list of associated nodes (adhoc) 
 *
 * wlan_if_t            : handle to the station vap
 * RETURNS:             : 
 */
void wlan_mlme_adhoc_assoc_list_build(wlan_if_t vaphandle);

/*
 * @MLME Free assoc list (adhoc)
 *
 * wlan_if_t            : handle to the station vap
 * RETURNS:             : 
 */
void wlan_mlme_adhoc_assoc_list_free(wlan_if_t vaphandle);

/*
 * @MLME Number of nodes in assoc list (adhoc)
 *
 * wlan_if_t            : handle to the station vap
 * RETURNS:             : 
 */
u_int32_t wlan_mlme_adhoc_assoc_list_count(wlan_if_t vaphandle);

/*
 * @MLME Return node at given index in assoc list (adhoc) 
 *
 * wlan_if_t            : handle to the station vap
 * u_int32_t            : index 
 * RETURNS:             : wlan_node_t
 */
wlan_node_t wlan_mlme_adhoc_assoc_list_get(wlan_if_t vaphandle, u_int32_t index);

/*
 * @sets the desired ssid list  .
 * ARGS :
 *  wlan_if_t       : handle to the vap .   
 *  nssid           : number of ssids passed in.
 *  ssidlist        : array of ssids.
 *
 *  RETURNS: 0 on success and -ve on failure.
 *           after this call is made, the ssid set remains permanent until the next call to the
 *           same function. 
 *           it always clears any ssid list set up before setting up new set.
 *           nssid =0 clears the previous setting.
 */
int wlan_set_desired_ssidlist(wlan_if_t vaphandle, u_int16_t nssid, ieee80211_ssid *ssidlist);

/*
 * @gets the desired ssid list  .
 * ARGS :
 *  wlan_if_t       : handle to the vap .   
 *  ssidlist        : array of ssids.
 *  nssid           : array length
 *
 *  RETURNS: number of ssids returned in the ssid list      
 */   
int wlan_get_desired_ssidlist(wlan_if_t vaphandle, ieee80211_ssid *ssidlist, int nssid);

/*
 * Public methods to set/get parameters used to build Candidate AP list.
 */

/*
 * @sets the desired ssid list  .
 * ARGS :
 *  wlan_if_t       : handle to the vap .   
 *  bnssid          : number of bssids passed in.
 *  bssidlist       : array of bssids.
 *
 *  RETURNS: 0 on success and -ve on failure.
 *           after this call is made, the bssid set remains permanent until the next call to the
 *           same function. 
 *           it always clears any bssid list set up before setting up new set.
 *           nbssid =0 clears the previous setting.
 */
int wlan_aplist_set_desired_bssidlist(wlan_if_t vaphandle, u_int16_t nbssid, u_int8_t (*bssidlist)[IEEE80211_ADDR_LEN]);

/*
 * @gets the number of desired bssid.
 * ARGS :
 *  wlan_if_t       : handle to the vap .   
 *
 *  RETURNS: number of bssids returned in the bssid list      
 */   
int wlan_aplist_get_desired_bssid_count(wlan_if_t vaphandle);

/*
 * @gets the desired bssid list  .
 * ARGS :
 *  wlan_if_t       : handle to the vap .   
 *  bssidlist       : array of bssids.
 *
 *  RETURNS: number of bssids returned in the bssid list      
 */   
int wlan_aplist_get_desired_bssidlist(wlan_if_t vaphandle, u_int8_t (*bssidlist)[IEEE80211_ADDR_LEN]);

/*
 * @gets one item from the desired bssid list  .
 * ARGS :
 *  wlan_if_t       : handle to the vap .   
 *  index           : the index of the list item to be returned.
 *  bssid           : a pointer to the variable where the address of the desired ssid
 *                    will be copied.
 *
 *  RETURNS: 0 if the index was valid and an address was returned, -ve otherwise.
 */   
int wlan_aplist_get_desired_bssid(wlan_if_t vaphandle, int index, u_int8_t **bssid);

/*
 * @sets the max age for entries in the scan list to be considered for association.
 *
 * wlan_if_t            : handle to the station vap
 * max_age              : maximum accepted age for scan entries, in ms.
 * RETURNS:             : IEEE80211 status code
 */
int wlan_aplist_set_max_age(wlan_if_t vaphandle, u_int32_t max_age);

/*
 * @gets the max age for entries in the scan list to be considered for association.
 *
 * wlan_if_t            : handle to the station vap
 * RETURNS:             : max age, in ms.
 */
u_int32_t wlan_aplist_get_max_age(wlan_if_t vaphandle);


/*
 * @sets the strict filtering flag. If this flag is set only entries updated during 
 *  the last scan will be considered for association.
 *
 * wlan_if_t            : handle to the station vap
 * strict_filtering     : the value to set
 * RETURNS:             : IEEE80211 status code
 */
bool wlan_aplist_set_strict_filtering(wlan_if_t vaphandle, bool strict_filtering);


/*
 * @gets the strict filtering flag.
 *
 * wlan_if_t            : handle to the station vap
 * RETURNS:             : strict filtering
 */
bool wlan_aplist_get_strict_filtering(wlan_if_t vaphandle);

/*
 * @sets the desired ssid list  .
 * ARGS :
 *  wlan_if_t       : handle to the vap .   
 *  nphy            : number of phy types passed in.
 *  phylist         : array of phy modes.
 *
 *  RETURNS: 0 on success and -ve on failure.
 *           after this call is made, the phy list set remains permanent until the next call to the
 *           same function. 
 *           it always clears any phy list set up before setting up new set.
 *           nphy =0 clears the previous setting.
 *            
 *   
 */
int wlan_set_desired_phylist(wlan_if_t vaphandle, enum ieee80211_phymode *phylist, u_int16_t nphy);

/*
 * @gets the desired phy list  .
 * ARGS :
 *  wlan_if_t       : handle to the vap .   
 *  phylist         : address of memory location where the address of the 
 *                    array of phy modes should be copied.
 *  nphy            : number of desired phy modes
 *  len             : passed-in array length
 *
 *  RETURNS: number of phy types returned in the phy list      
 */   

int wlan_get_desired_phylist(wlan_if_t vaphandle, enum ieee80211_phymode *phylist, u_int16_t *nphy, u_int16_t len);

/*
 * @sets the desired BSS type
 *
 * wlan_if_t            : handle to the station vap
 * bss_type             : desired BSS type.
 * RETURNS:             : IEEE80211 status code
 */
int wlan_aplist_set_desired_bsstype(wlan_if_t vaphandle, enum ieee80211_opmode bss_type);

/*
 * @gets the value of the "accept any bssid" flag .
 *
 * wlan_if_t            : handle to the station vap
 * RETURNS:             : true if the flag is set, false otherwise.
 */
bool wlan_aplist_get_accept_any_bssid(wlan_if_t vaphandle);

/*
 * @sets the value of the "ignore all mac addresses" flag. If this flag is set the STA will not
 *  associate to any AP.
 *
 * wlan_if_t            : handle to the station vap
 * flag                 : true/false
 * RETURNS:             : 0 if the flag was set correctly, -ve otherwise.
 */
int wlan_aplist_set_ignore_all_mac_addresses(wlan_if_t vaphandle, bool flag);

/*
 * @gets the value of the "ignore all mac addresses" flag. If this flag is set the STA will not
 *  associate to any AP.
 *
 * wlan_if_t            : handle to the station vap
 * RETURNS:             : true if the flag is set, false otherwise.
 */
bool wlan_aplist_get_ignore_all_mac_addresses(wlan_if_t vaphandle);

/*
 * @sets the list of excluded MAC addresses. APs with addresses in this list will not be 
 *  considered for association.
 * ARGS :
 *  wlan_if_t       : handle to the vap .   
 *  n_entries       : number of addresses passed in.
 *  macaddress      : array of MAC addresses.
 *
 *  RETURNS: 0 on success and -ve on failure.
 *           after this call is made, the MAC address list remains permanent until the next call to the
 *           same function. Any addresses previously in the list are removed.
 *           n_entries=0 clears the previous setting.
 */
int wlan_aplist_set_exc_macaddresslist(wlan_if_t vaphandle, 
                                       u_int16_t n_entries, 
                                       u_int8_t  (*macaddress)[IEEE80211_ADDR_LEN]);

/*
 * @gets the list of excluded MAC addresses. APs with addresses in this list will not be 
 *  considered for association.
 * ARGS :
 *  wlan_if_t       : handle to the vap .   
 *  macaddress      : array of MAC addresses.
 *
 *  RETURNS: number of entries in the list.
 */
int wlan_aplist_get_exc_macaddresslist(wlan_if_t vaphandle, 
                                       u_int8_t  (*macaddress)[IEEE80211_ADDR_LEN]);

/*
 * @gets the number of entries in the list of excluded MAC addresses. 
 * ARGS :
 *  wlan_if_t       : handle to the vap .   
 *
 *  RETURNS: number of entries in the list.
 */
int wlan_aplist_get_exc_macaddress_count(wlan_if_t vaphandle);

/*
 * @gets one entry from the list of excluded MAC addresses. 
 * ARGS :
 *  wlan_if_t       : handle to the vap .   
 *  index           : the index of the list item to be returned.
 *  pmacaddress     : a pointer to the variable where the address will be copied.
 *
 *  RETURNS: 0 if the index was valid and an address was returned, -ve otherwise.
 */
int wlan_aplist_get_exc_macaddress(wlan_if_t vaphandle, int index, u_int8_t **pmacaddress);

/*
 * @MLME get the desired BSS type
 *
 * wlan_if_t            : handle to the station vap
 * RETURNS:             : desired BSS type.
 */
enum ieee80211_opmode wlan_aplist_get_desired_bsstype(wlan_if_t vaphandle);

/*
 * @MLME set the preferred SuperAG status
 *
 * wlan_if_t            : handle to the station vap
 * pref_super_ag_status : preferred SuperAG status.
 * RETURNS:             : IEEE80211 status code
 */
int wlan_aplist_set_pref_super_ag_status(wlan_if_t vaphandle, int pref_super_ag_status);

/*
 * @MLME get the preferred SuperAG status
 *
 * wlan_if_t            : handle to the station vap
 * RETURNS:             : preferred SuperAG status.
 */
int wlan_aplist_get_pref_super_ag_status(wlan_if_t vaphandle);

/*
 * @MLME set the Tx Power Delta
 *
 * wlan_if_t            : handle to the station vap
 * tx_power_delta       : Tx Power Delta.
 * RETURNS:             : IEEE80211 status code
 */
int wlan_aplist_set_tx_power_delta(wlan_if_t vaphandle, int tx_power_delta);

/*
 * @MLME resets all aplist parameters to their default values.
 *
 * wlan_if_t            : handle to the station vap
 */
void wlan_aplist_init(wlan_if_t vaphandle);

/*
 * Public methods to build/free and query Candidate AP list.
 */

/*
 * @Build list of candidate APs.
 *
 * wlan_if_t            : handle to the station vap
 * strict_filtering     : true indicates entries older than the most recent scan are allowed,
 *                        false indicates otherwise
 * active_ap            : pointer to the active (home) AP.
 *
 *     This function uses the current list of association parameters to build the candidate list.
 */
void wlan_candidate_list_build(wlan_if_t         vaphandle, 
                               bool              strict_filtering, 
                               wlan_scan_entry_t active_ap);

/*
 * @Clear list candidate APs
 *
 * wlan_if_t            : handle to the station vap
 */
void wlan_candidate_list_free(wlan_if_t vaphandle);

/*
 * @Number of entries in the candidate AP list.
 *
 * wlan_if_t            : handle to the station vap
 * RETURNS:             : Number of entries in the candidate AP list
 */
int wlan_candidate_list_count(wlan_if_t vaphandle);

/*
 * @Retrieve a candidate AP.
 *
 * wlan_if_t            : handle to the station vap
 * index                : the index of the candidate AP to be returned. Valid values are 
 *                        in the range 0-wlan_candidate_list_count().
 * RETURNS:             : A candidate AP.
 */
wlan_scan_entry_t wlan_candidate_list_get(wlan_if_t vaphandle, int index);

/*
 * @Get general statistics
 * ARGS :
 *  wlan_if_t           : vap handle
 * RETURNS:             : pointer to vap statistics.
 */
const struct ieee80211_stats *wlan_gen_stats(wlan_if_t vaphandle);

/*
 * @Get MAC statistics
 * ARGS :
 *  wlan_if_t           : vap handle
 *  is_mcast            : if set, get multicast MAC statistics,
 *                        otherwise, get unicast MAC statistics
 * RETURNS:             : pointer to MAC statistics.
 */
const struct ieee80211_mac_stats *wlan_mac_stats(wlan_if_t vaphandle, int is_mcast);

/*
 * @Get PHY statistics
 * ARGS :
 *  wlan_dev_t          : dev handle
 *  mode                : the intended PHY mode.
 * RETURNS:             : pointer to PHY statistics of phymode.
 */
const struct ieee80211_phy_stats *wlan_phy_stats(wlan_dev_t devhandle, enum ieee80211_phymode mode);

/*
 * @Get timestamp of the last directed frame received or transmitted.
 * ARGS :
 *  wlan_dev_t          : dev handle
 * RETURNS:             : timestamp of the last directed frame.
 */
systime_t wlan_get_directed_frame_timestamp(wlan_if_t vaphandle);

/*
 * @Get timestamp of the data frame containing actual data that was received or
 *      transmitted. Null data frames are not counted. This is necessary for 
 *      modules such as SCAN or LED which depend on data load.
 * ARGS :
 *  wlan_dev_t          : dev handle
 * RETURNS:             : timestamp of the last directed frame.
 */
systime_t wlan_get_traffic_indication_timestamp(wlan_if_t vaphandle);

/*
 * @Indicates whether station is in connected state.
 * ARGS :
 *  wlan_dev_t          : dev handle
 * RETURNS:             : timestamp of the last directed frame.
 */
bool wlan_is_connected(wlan_if_t vaphandle);

/*
 * @send ADD TSPEC action frame to the destination.
 * ARGS:
 *  wlan_if_t           : vap handle
 *  macaddr             : peer mac address
 *  tsinfo              : TSPEC info
 */
int wlan_send_addts(wlan_if_t vaphandle, u_int8_t *macaddr, ieee80211_tspec_info *tsinfo);

/*
 * @send DEL TSPEC action frame to the destination.
 * ARGS:
 *  wlan_if_t           : vap handle
 *  macaddr             : peer mac address
 *  tsinfo              : TSPEC info
 */
int wlan_send_delts(wlan_if_t vaphandle, u_int8_t *macaddr, ieee80211_tspec_info *tsinfo);

/*
 * @start a reset: it will flush all the pending packets, disable the interrupt and put
 * hardware in a safe state for reset.
 * ARGS:
 *  wlan_if_t           : vap handle
 *  reset_req           : reset request
 */
int wlan_reset_start(wlan_if_t vaphandle, ieee80211_reset_request *reset_req);

/*
 * @reset
 * ARGS:
 *  wlan_if_t           : vap handle
 *  reset_req           : reset request
 */
int wlan_reset(wlan_if_t vaphandle, ieee80211_reset_request *reset_req);

/*
 * @complete a reset: it will restore the interrupt and put hardware in a good state for tx/rx.
 * ARGS:
 *  wlan_if_t           : vap handle
 *  reset_req           : reset request
 */
int wlan_reset_end(wlan_if_t vaphandle, ieee80211_reset_request *reset_req);

#endif /* _IEEE80211_API_H_ */
