/*
 *  Copyright (c) 2008 Atheros Communications Inc.  All rights reserved.
 */

#ifndef _IEEE80211_DEFINES_H_
#define _IEEE80211_DEFINES_H_

/*
 * Public defines for Atheros Upper MAC Layer
 */

/**
 * @brief Opaque handle of 802.11 protocal layer.
 */
struct ieee80211com;
typedef struct ieee80211com *wlan_dev_t; 

/**
 * @brief Opaque handle of network instance (vap) in 802.11 protocal layer.
*/
struct ieee80211vap;
typedef struct ieee80211vap *wlan_if_t; 

/**
 * @brief Opaque handle of a node in the wifi network.
 */
struct ieee80211_node;
typedef struct ieee80211_node *wlan_node_t;

/**
 * @brief Opaque handle of OS interface (ifp in the case of unix ).
 */
struct _os_if_t;
typedef struct _os_if_t *os_if_t; 

/**
 * @brief Opaque handle of a channel.
 */
struct ieee80211_channel;
typedef struct ieee80211_channel *wlan_chan_t;

/**
 * @brief Opaque handle scan_entry.
 */
struct ieee80211_scan_entry; 
typedef struct ieee80211_scan_entry *wlan_scan_entry_t;

typedef struct _ieee80211_ssid {
    int         len;
    u_int8_t    ssid[IEEE80211_NWID_LEN];
} ieee80211_ssid;

typedef struct ieee80211_tx_status {
    int         ts_flags;
#define IEEE80211_TX_ERROR          0x01
#define IEEE80211_TX_XRETRY         0x02

    int         ts_retries;     /* number of retries to successfully transmit this frame */
} ieee80211_xmit_status;

typedef struct ieee80211_rx_status {
    int         rs_flags;
#define IEEE80211_RX_FCS_ERROR      0x01
#define IEEE80211_RX_MIC_ERROR      0x02
#define IEEE80211_RX_DECRYPT_ERROR  0x04

    int         rs_rssi;       /* RSSI (noise floor ajusted) */
    int         rs_abs_rssi;   /* absolute RSSI */
    int         rs_datarate;   /* data rate received */

    enum ieee80211_phymode rs_phymode;
    int         rs_freq;
    
    union {
        u_int8_t            data[8];
        u_int64_t           tsf;
    } rs_tstamp;
} ieee80211_recv_status;

/*
 * flags to be passed to ieee80211_vap_create function .
 */
#define IEEE80211_CLONE_BSSID           0x0001  /* allocate unique mac/bssid */
#define IEEE80211_CLONE_NOBEACONS       0x0002  /* don't setup beacon timers */
#define IEEE80211_CLONE_WDS             0x0004  /* enable WDS processing */
#define IEEE80211_CLONE_WDSLEGACY       0x0008  /* legacy WDS operation */

typedef int IEEE80211_REASON_CODE;
typedef int IEEE80211_STATUS;

typedef struct _wlan_event_handler_table {
    void (*wlan_receive) (os_if_t osif, wbuf_t wbuf,
                          u_int16_t type, u_int16_t subtype,
                          ieee80211_recv_status *rs);                       /* callback to receive all the packets */
    int  (*wlan_receive_filter_80211) (os_if_t osif, wbuf_t wbuf,
                                       u_int16_t type, u_int16_t subtype,
                                       ieee80211_recv_status *rs);          /* callback to receive and filter all the 802.11 frames*/
    void (*wlan_receive_monitor_80211) (os_if_t osif, wbuf_t wbuf,
                                        ieee80211_recv_status *rs);         /* recieve 802.11 frames for monitor mode only*/
    int  (*wlan_dev_xmit_queue) (os_if_t osif, wbuf_t wbuf);                /* queue packet to the device for transmit */
    void (*wlan_vap_xmit_queue) (os_if_t osif, wbuf_t wbuf);                /* queue packet to the interface for transmit */
    void (*wlan_xmit_update_status)(os_if_t osif, wbuf_t wbuf,
                                    ieee80211_xmit_status *ts);             /* callback to indicate tx completion.
                                                                             * NB: this is to allow IHV to update certain statistics,
                                                                             * it should NOT consume the wbuf */
} wlan_event_handler_table;

typedef struct _wlan_mlme_event_handler_table {
    /* MLME confirmation handler */
    void (*mlme_join_complete_infra)(os_if_t, IEEE80211_STATUS);
    void (*mlme_join_complete_adhoc)(os_if_t, IEEE80211_STATUS);
    void (*mlme_auth_complete)(os_if_t, IEEE80211_STATUS);
    void (*mlme_assoc_req)(os_if_t, wbuf_t wbuf);
    void (*mlme_assoc_complete)(os_if_t, IEEE80211_STATUS, u_int16_t aid, wbuf_t wbuf);
    void (*mlme_reassoc_complete)(os_if_t, IEEE80211_STATUS, u_int16_t aid, wbuf_t wbuf);
    void (*mlme_deauth_complete)(os_if_t, IEEE80211_STATUS);
    void (*mlme_disassoc_complete)(os_if_t, IEEE80211_STATUS);

    /* MLME indication handler */
    void (*mlme_auth_indication)(os_if_t, u_int8_t *macaddr, u_int16_t result);
    void (*mlme_deauth_indication)(os_if_t, u_int8_t *macaddr, u_int16_t reason_code);
    void (*mlme_assoc_indication)(os_if_t, u_int8_t *macaddr, u_int16_t result, wbuf_t wbuf, wbuf_t resp_wbuf);
    void (*mlme_reassoc_indication)(os_if_t, u_int8_t *macaddr, u_int16_t result, wbuf_t wbuf, wbuf_t resp_wbuf);
    void (*mlme_disassoc_indication)(os_if_t, u_int8_t *macaddr, u_int16_t reason_code);
    void (*mlme_ibss_merge_start_indication)(os_if_t, u_int8_t *bssid);
    void (*mlme_ibss_merge_completion_indication)(os_if_t, u_int8_t *bssid);
} wlan_mlme_event_handler_table;

typedef struct _wlan_misc_event_handler_table {
    void (*wlan_channel_change) (os_if_t, wlan_chan_t chan);
    void (*wlan_country_changed) (os_if_t, char *country);
    void (*wlan_linkspeed)(os_if_t, u_int32_t linkspeed);
    void (*wlan_michael_failure_indication)(os_if_t, const u_int8_t *frm, u_int keyix);
    void (*wlan_replay_failure_indication)(os_if_t, const u_int8_t *frm, u_int keyix);
    void (*wlan_beacon_miss_indication)(os_if_t);
    void (*wlan_device_error_indication)(os_if_t);
} wlan_misc_event_handler_table;

/* 
 *                        
 *  wlan_dev_xmit      : the OS dependent layer needs to register a callback function.
 *                       umac layer uses this callback to xmit a packet. the implementaion
 *                       of this  call back function would implement OS specific handling of 
 *                       the wbuf and at the end it if it decides to xmit the wbuf it should 
 *                       call the ath_tx_send (the ath layer xmit funtion). the implementation
 *                       should return 0 on success and -ve value on failure.
 *
 *  wlan_vap_xmit      : the OS dependent layer needs to register a callback function.
 *                       umac layer uses this callback to xmit a packet. the implementaion
 *                       of this  call back function would implement send the packet on to the 
 *                       transmit queue of the vap. if IEEE80211_DELIVER_80211 param is set the 
 *                       frame is 802.11 frame else  it is ethernet (802.3) frame.
 * 
 *  wlan_receive       : handler to receive all the packets icluding data,mgmt,control.
 *                       UMAC will deliver standard 802.11 frames (with qos control removed)
 *                       if IEEE80211_DELIVER_80211 param is set. 
 *                       UMAC will deliver ethernet frames (with 802.11 header decapped)
 *                       if IEEE80211_DELIVER_80211 param is not set. 
 *                       the handler should cosume the wbuf (release it eventually). 
 *
 *  wlan_receive_filter_80211 : handler to receive all the frames with 802.11 header unstripped but the FCS stripped at the end.
 *                              implementation of the handler should make a copy of the wbuf and should not
 *                              release the wbuf passed via the handler. implementation should return non zero if the packet
 *                              should not be processed further by umac.
 *
 *  wlan_receive_monitor_80211 : handler to receive the  all the frames with the 802.11 header including the FCS.
 *                               the handler should cosume the wbuf (release it eventually). 
 *                               this function is exclusively used for monitor mode.
 *
 *  
 */

/*
 * scan API related structs.
 */
typedef enum _ieee80211_scan_type {
    IEEE80211_SCAN_BACKGROUND,
    IEEE80211_SCAN_FOREGROUND
} ieee80211_scan_type;

typedef enum _ieee80211_scan_priority {
    IEEE80211_SCAN_PRIORITY_LOW,
    IEEE80211_SCAN_PRIORITY_HIGH
} ieee80211_scan_priority;

#define IEEE80211_SCAN_PASSIVE      0x01 /* passively scan all the channels */
#define IEEE80211_SCAN_ACTIVE       0x02 /* actively  scan all the channels (regdomain rules still apply) */
#define IEEE80211_SCAN_2GHZ         0x04 /* scan 2GHz band */
#define IEEE80211_SCAN_5GHZ         0x08 /* scan 5GHz band */
#define IEEE80211_SCAN_ALLBANDS     (IEEE80211_SCAN_5GHZ | IEEE80211_SCAN_2GHZ)
#define IEEE80211_SCAN_CONTINUOUS   0x10 /* keep scanning until maxscantime expires */
#define IEEE80211_SCAN_FORCED       0x20 /* forced scan (OS request) - should proceed even in the presence of data traffic */
#define IEEE80211_SCAN_NOW          0x40 /* scan now (User request)  - should proceed even in the presence of data traffic */

#define IEEE80211_SCAN_PARAMS_MAX_SSID     10
#define IEEE80211_SCAN_PARAMS_MAX_BSSID    10

typedef struct _ieee80211_scan_params {
    ieee80211_scan_type    type;
    int                    min_dwell_time;   /* min time in msec on the channel */
    int                    max_dwell_time;   /* max time in msec on the channel (if no repsonse) */
    int                    min_rest_time;    /* min time in msec on the BSS channel, only valid for BG scan */
    int                    max_rest_time;    /* max time in msec on the BSS channel, only valid for BG scan */
    int                    idle_time;        /* time in msec on bss channel before switching channel */ 
    int                    max_scan_time;    /* maximum time in msec allowed for scan  */ 
    int                    probe_delay;      /* delay in msec before sending probe request */
    int                    min_beacon_count; /* number of home AP beacons to receive before leaving the home channel */
    int                    beacon_timeout;   /* maximum time to wait for beacons */
    int                    flags;            /* scan flags */
    int                    num_channels;     /* number of channels to scan */ 
    u_int32_t              *chan_list;       /* array of ieee channels (or) frequencies to scan */
    int                    num_ssid;         /* number of desired ssids */ 
    ieee80211_ssid         ssid_list[IEEE80211_SCAN_PARAMS_MAX_SSID];
    int                    num_bssid;
    u_int8_t               bssid_list[IEEE80211_SCAN_PARAMS_MAX_BSSID][IEEE80211_ADDR_LEN];
    struct ieee80211_node  *bss_node;        /* BSS node */
    int                    ie_len;           /* length of the ie data to be added to probe req */ 
    u_int8_t               *ie_data;         /* pointer to ie data */
} ieee80211_scan_params;

/**************************************
 * Called before attempting to roam.  Modifies the rssiAdder of a BSS
 * based on the preferred status of a BSS.
 *
 * According to CCX spec, AP in the neighbor list is not meant for giving extra
 * weightage in roaming. By doing so, roaming becomes sticky. See bug 21220.
 * Change the weightage to 0. Cisco may ask in future for a user control of
 * this weightage.
 */
#define PREFERRED_BSS_RANK                20
#define NEIGHBOR_BSS_RANK                  0    /* must be less than preferred BSS rank */

/*
 * The utility of the BSS is the metric used in the selection
 * of a BSS. The Utility of the BSS is reduced if we just left the BSS.
 * The Utility of the BSS is not reduced if we have left the
 * BSS for 8 seconds (8000ms) or more.
 * 2^13 milliseconds is a close approximation to avoid expensive division
 */
#define LAST_ASSOC_TIME_DELTA_REQUIREMENT (1 << 13) // 8192

#define QBSS_SCALE_MAX                   255  /* Qbss channel load Max value */
#define QBSS_SCALE_DOWN_FACTOR             2  /* scale factor to reduce Qbss channel load */
#define QBSS_HYST_ADJ                     60  /* Qbss Weightage factor for the current AP */

/*
 * Flags used to set field APState
 */
#define AP_STATE_GOOD    0x00
#define AP_STATE_BAD     0x01
#define AP_STATE_RETRY   0x10

/*
 * idle time is only valid for scan type IEEE80211_SCAN_BACKGROUND.
 * if idle time is set then the scanner would change channel from BSS
 * channel to foreign channel only if both resttime is expired and 
 * the theres was not traffic for idletime msec on the bss channel.
 * value of 0 for idletime would cause the channel to switch from BSS
 * channel to foreign channel as soon  as the resttime is expired.
 *
 * if maxscantime is nonzero and if the scanner can not complete the 
 * scan in maxscantime msec then the scanner will cancel the scan and
 * post IEEE80211_SCAN_COMPLETED event with reason SCAN_TIMEDOUT.
 *
 */  

/*
 * chanlist can be either ieee channels (or) frequencies.
 * if a value is less than 1000 implementation assumes it 
 * as ieee channel # otherwise implementation assumes it 
 * as frequency in Mhz.
 */ 

typedef enum _ieee80211_scan_event_type {
    IEEE80211_SCAN_STARTED,
    IEEE80211_SCAN_COMPLETED,
    IEEE80211_SCAN_PAUSED,
    IEEE80211_SCAN_RESUMED,
    IEEE80211_SCAN_RESTARTED,
    IEEE80211_SCAN_HOME_CHANNEL,
    IEEE80211_SCAN_FOREIGN_CHANNEL,
    IEEE80211_SCAN_BSSID_MATCH 
} ieee80211_scan_event_type;

typedef enum ieee80211_scan_completion_reason {
    IEEE80211_REASON_NONE,
    IEEE80211_REASON_COMPLETED,
    IEEE80211_REASON_CANCELLED,
    IEEE80211_REASON_TIMEDOUT,
} ieee80211_scan_completion_reason;

typedef struct _ieee80211_scan_event {
    ieee80211_scan_event_type           type;
    ieee80211_scan_completion_reason    reason;
    wlan_chan_t                         chan;
    u_int16_t                           req_id; /* Request ID passed to the scan_start function */
    wlan_scan_entry_t                   scan_entry;
} ieee80211_scan_event;

/*
 * the sentry field of tht ieee80211_scan_event is only valid if the
 * event type is IEEE80211_SCAN_BSSID_MATCH.
 */

typedef void (*ieee80211_scan_event_handler) (wlan_if_t vaphandle, ieee80211_scan_event *event, void *arg);  

typedef struct _ieee80211_scan_info {
    ieee80211_scan_type    type;
    u_int16_t              req_id;           /* originator ID passed to the scan_start function */
    int                    min_dwell_time;   /* min time in msec on the channel */
    int                    max_dwell_time;   /* max time in msec on the channel (if no repsonse) */
    int                    min_rest_time;    /* min time in msec on the BSS channel, only valid for BG scan */
    int                    max_rest_time;    /* max time in msec on the BSS channel, only valid for BG scan */
    int                    min_beacon_count; /* number of home AP beacons to receive before leaving the home channel */
    int                    flags;            /* scan flags */
    systime_t              scan_start_time;  /* system time when last scani started */ 
    int                    in_progress : 1,  /* if the scan is in progress */
                           cancelled   : 1;  /* if the scan is cancelled */
} ieee80211_scan_info;

#define MAX_CHAINS 3

typedef struct _wlan_rssi_info {
    u_int8_t    avg_rssi;     /* average rssi */
    u_int8_t    valid_mask;   /* bitmap of valid elements in rssi_ctrl/ext array */
    u_int8_t    rssi_ctrl[MAX_CHAINS];
    u_int8_t    rssi_ext[MAX_CHAINS];
} wlan_rssi_info;

typedef enum _wlan_rssi_type {
    WLAN_RSSI_TX,
    WLAN_RSSI_RX,
    WLAN_RSSI_BEACON  /* rssi of the beacon, only valid for STA/IBSS vap */
} wlan_rssi_type;

typedef enum _ieee80211_rate_type {
    IEEE80211_RATE_TYPE_LEGACY,
    IEEE80211_RATE_TYPE_MCS20,
    IEEE80211_RATE_TYPE_MCS40,
} ieee80211_rate_type;

typedef struct _ieee80211_rate_info {
    ieee80211_rate_type    type;
    u_int32_t              rate;  /* rate in kbps */
    u_int32_t              mcs;   /* mcs index . is valid if rate type is MCS20 or MCS40 */
} ieee80211_rate_info;

typedef enum _ieee80211_node_param_type {
    IEEE80211_NODE_PARAM_TX_POWER,
    IEEE80211_NODE_PARAM_ASSOCID,
    IEEE80211_NODE_PARAM_INACT,     /* inactivity timer value */
    IEEE80211_NODE_PARAM_AUTH_MODE, /* auth mode */
    IEEE80211_NODE_PARAM_CAP_INFO,  /* auth mode */
} ieee80211_node_param_type;

/*
 * Per/node (station) statistics available when operating as an AP.
 */
struct ieee80211_nodestats {
    u_int32_t    ns_rx_data;             /* rx data frames */
    u_int32_t    ns_rx_mgmt;             /* rx management frames */
    u_int32_t    ns_rx_ctrl;             /* rx control frames */
    u_int32_t    ns_rx_ucast;            /* rx unicast frames */
    u_int32_t    ns_rx_mcast;            /* rx multi/broadcast frames */
    u_int64_t    ns_rx_bytes;            /* rx data count (bytes) */
    u_int64_t    ns_rx_beacons;          /* rx beacon frames */
    u_int32_t    ns_rx_proberesp;        /* rx probe response frames */

    u_int32_t    ns_rx_dup;              /* rx discard 'cuz dup */
    u_int32_t    ns_rx_noprivacy;        /* rx w/ wep but privacy off */
    u_int32_t    ns_rx_wepfail;          /* rx wep processing failed */
    u_int32_t    ns_rx_demicfail;        /* rx demic failed */
    u_int32_t    ns_rx_decap;            /* rx decapsulation failed */
    u_int32_t    ns_rx_defrag;           /* rx defragmentation failed */
    u_int32_t    ns_rx_disassoc;         /* rx disassociation */
    u_int32_t    ns_rx_deauth;           /* rx deauthentication */
    u_int32_t    ns_rx_action;           /* rx action */
    u_int32_t    ns_rx_decryptcrc;       /* rx decrypt failed on crc */
    u_int32_t    ns_rx_unauth;           /* rx on unauthorized port */
    u_int32_t    ns_rx_unencrypted;      /* rx unecrypted w/ privacy */

    u_int32_t    ns_tx_data;             /* tx data frames */
    u_int32_t    ns_tx_mgmt;             /* tx management frames */
    u_int32_t    ns_tx_ucast;            /* tx unicast frames */
    u_int32_t    ns_tx_mcast;            /* tx multi/broadcast frames */
    u_int64_t    ns_tx_bytes;            /* tx data count (bytes) */
    u_int32_t    ns_tx_probereq;         /* tx probe request frames */
    u_int32_t    ns_tx_uapsd;            /* tx on uapsd queue */

    u_int32_t    ns_tx_novlantag;        /* tx discard 'cuz no tag */
    u_int32_t    ns_tx_vlanmismatch;     /* tx discard 'cuz bad tag */

    u_int32_t    ns_tx_eosplost;         /* uapsd EOSP retried out */

    u_int32_t    ns_ps_discard;          /* ps discard 'cuz of age */

    u_int32_t    ns_uapsd_triggers;      /* uapsd triggers */

    /* MIB-related state */
    u_int32_t    ns_tx_assoc;            /* [re]associations */
    u_int32_t    ns_tx_assoc_fail;       /* [re]association failures */
    u_int32_t    ns_tx_auth;             /* [re]authentications */
    u_int32_t    ns_tx_auth_fail;        /* [re]authentication failures*/
    u_int32_t    ns_tx_deauth;           /* deauthentications */
    u_int32_t    ns_tx_deauth_code;      /* last deauth reason */
    u_int32_t    ns_tx_disassoc;         /* disassociations */
    u_int32_t    ns_tx_disassoc_code;    /* last disassociation reason */
    u_int32_t    ns_psq_drops;           /* power save queue drops */
};

/*
 * station power save mode.
 */
typedef enum ieee80211_psmode {
    IEEE80211_PWRSAVE_NONE = 0,          /* no power save */
    IEEE80211_PWRSAVE_LOW,
    IEEE80211_PWRSAVE_NORMAL,
    IEEE80211_PWRSAVE_MAXIMUM
} ieee80211_pwrsave_mode;

/*
 * apps power save state.
 */
typedef enum {
     APPS_AWAKE = 0,
     APPS_PENDING_SLEEP,
     APPS_SLEEP,
     APPS_FAKE_SLEEP,           /* Pending blocking sleep */
     APPS_FAKING_SLEEP,         /* Blocking sleep */
     APPS_UNKNOWN_PWRSAVE,
} ieee80211_apps_pwrsave_state;

typedef enum _iee80211_mimo_powersave_mode {
    IEEE80211_MIMO_POWERSAVE_NONE,    /* no mimo power save */
    IEEE80211_MIMO_POWERSAVE_STATIC,  /* static mimo power save */
    IEEE80211_MIMO_POWERSAVE_DYNAMIC  /* dynamic mimo powersave */
} ieee80211_mimo_powersave_mode;

#ifdef ATH_COALESCING
typedef enum _ieee80211_coalescing_state {
    IEEE80211_COALESCING_DISABLED   = 0,        /* Coalescing is disabled*/
    IEEE80211_COALESCING_DYNAMIC    = 1,        /* Dynamically move to Enabled state based on Uruns*/
    IEEE80211_COALESCING_ENABLED    = 2,        /* Coalescing is enabled*/
} ieee80211_coalescing_state;

#define IEEE80211_TX_COALESCING_THRESHOLD     5 /* Number of underrun errors to trigger coalescing */
#endif

typedef enum _ieee80211_cap {
    IEEE80211_CAP_SHSLOT,                    /* CAPABILITY: short slot */
    IEEE80211_CAP_SHPREAMBLE,                /* CAPABILITY: short premable */
    IEEE80211_CAP_MULTI_DOMAIN,              /* CAPABILITY: multiple domain */
    IEEE80211_CAP_WMM,                       /* CAPABILITY: WMM */
    IEEE80211_CAP_SUPERAG,                   /* CAPABILITY: super a/g */
    IEEE80211_CAP_HT,                        /* CAPABILITY: HT */
} ieee80211_cap;

typedef enum _ieee80211__device_param {
    IEEE80211_DEVICE_TX_CHAIN_MASK,
    IEEE80211_DEVICE_RX_CHAIN_MASK,
    IEEE80211_DEVICE_PROTECTION_MODE,               /* protection mode*/
} ieee80211_device_param;

typedef enum _ieee80211_param {
    IEEE80211_BEACON_INTVAL,                 /* in TUs */
    IEEE80211_LISTEN_INTVAL,                 /* number of beacons */
    IEEE80211_DTIM_INTVAL,                   /* number of beacons */
    IEEE80211_ATIM_WINDOW,                   /* ATIM window */
    IEEE80211_SHORT_SLOT,                    /* short slot on/off */
    IEEE80211_SHORT_PREAMBLE,                /* short preamble on/off */
    IEEE80211_RTS_THRESHOLD,                 /* rts threshold, 0 means no rts threshold  */
    IEEE80211_FRAG_THRESHOLD,                /* fragmentation threshold, 0 means no rts threshold  */
    IEEE80211_FIXED_RATE,                    /* fixed rate  0-15 : MCS indices.  250 to 54000 : rate in Kbps , 0xffffffff: auto rate  */
    IEEE80211_MCAST_RATE,                    /* rate in Kbps */
    IEEE80211_TXPOWER,                       /* in 0.5db units */
    IEEE80211_AMPDU_DENCITY,                 /* AMPDU dencity*/
    IEEE80211_AMPDU_LIMIT,                   /* AMPDU limit*/
    IEEE80211_WPS_MODE,                      /* WPS mode*/
    IEEE80211_TSN_MODE,                      /* TSN mode*/
    IEEE80211_MULTI_DOMAIN,                  /* Multiple domain */
    IEEE80211_SAFE_MODE,                     /* Safe mode */
    IEEE80211_PERSTA_KEYTABLE_SIZE,          /* IBSS-only, read-only: persta key table size */
    IEEE80211_RECEIVE_80211,                 /* deliver std 802.11 frames 802.11 instead of ethernet frames on the rx */
    IEEE80211_SEND_80211,                    /* OS sends std 802.11 frames 802.11 instead of ethernet frames on tx side */
    IEEE80211_MSG_FLAGS,                     /* flags to turn on different debug messages */
                                             /* features. 0:feature is off. 1:feature is on. */
    IEEE80211_FEATURE_WMM,                   /* WMM */
    IEEE80211_FEATURE_WMM_PWRSAVE,           /* WMM Power Save */
    IEEE80211_FEATURE_UAPSD,                 /* UAPSD setting (BE/BK/VI/VO) */
    IEEE80211_FEATURE_WDS,                   /* dynamic WDS feature */
    IEEE80211_FEATURE_PRIVACY,               /* encryption */
    IEEE80211_FEATURE_DROP_UNENC,            /* drop un encrypted frames */
    IEEE80211_FEATURE_COUNTER_MEASURES ,     /* turn on couter measures */
    IEEE80211_FEATURE_HIDE_SSID,             /* turn on hide ssid feature */
    IEEE80211_FEATURE_APBRIDGE,              /* turn on internal mcast traffic bridging for AP */
    IEEE80211_FEATURE_PUREB,                 /* turn on pure B mode for AP */
    IEEE80211_FEATURE_PUREG,                 /* turn on pure G mode for AP */
    IEEE80211_FEATURE_SUPERAG,               /* turn on super A/G support */
    IEEE80211_FEATURE_REGCLASS,              /* add regulatory class IE in AP */
    IEEE80211_FEATURE_COUNTRY_IE,            /* add country IE in AP */
    IEEE80211_FEATURE_DOTH,                  /* enable 802.11h */
    IEEE80211_FEATURE_PURE11N,               /* enable pure 11n  mode */
    IEEE80211_FEATURE_PRIVATE_RSNIE,         /* enable OS shim to setup RSN IE*/
#ifdef ATH_COALESCING
    IEEE80211_FEATURE_TX_COALESCING,         /* enable tx coalescing */
#endif
} ieee80211_param;

#define  IEEE80211_PROTECTION_NONE 0
#define  IEEE80211_PROTECTION_CTSTOSELF 1
#define  IEEE80211_PROTECTION_RTS_CTS 2

typedef enum _ieee80211_privacy_filter {
    IEEE80211_PRIVACY_FILTER_ALLWAYS,
    IEEE80211_PRIVACY_FILTER_KEY_UNAVAILABLE,
} ieee80211_privacy_filter ;

typedef enum _ieee80211_privacy_filter_packet_type {
    IEEE80211_PRIVACY_FILTER_PACKET_UNICAST,
    IEEE80211_PRIVACY_FILTER_PACKET_MULTICAST,
    IEEE80211_PRIVACY_FILTER_PACKET_BOTH
} ieee80211_privacy_filter_packet_type ;

typedef struct _ieee80211_privacy_excemption_filter {
    u_int16_t  ether_type; /* type of ethernet to apply this filter, in host byte order*/
    ieee80211_privacy_filter      filter_type; 
    ieee80211_privacy_filter_packet_type  packet_type;
} ieee80211_privacy_exemption;

/*
 * Authentication mode.
 * NB: the usage of auth modes NONE, AUTO are deprecated,
 * they are implemented through combinations of other auth modes
 * and cipher types. The deprecated values are preserved here to
 * maintain binary compatibility with applications like
 * wpa_supplicant and hostapd.
 */
typedef enum _ieee80211_auth_mode {
    IEEE80211_AUTH_NONE     = 0, /* deprecated */
    IEEE80211_AUTH_OPEN     = 1, /* open */
    IEEE80211_AUTH_SHARED   = 2, /* shared-key */
    IEEE80211_AUTH_8021X    = 3, /* 802.1x */
    IEEE80211_AUTH_AUTO     = 4, /* deprecated */
    IEEE80211_AUTH_WPA      = 5, /* WPA */
    IEEE80211_AUTH_RSNA     = 6, /* WPA2/RSNA */
    IEEE80211_AUTH_CCKM     = 7, /* CCK */
} ieee80211_auth_mode;

#define IEEE80211_AUTH_MAX      (IEEE80211_AUTH_CCKM+1)

/*
 * Cipher types.
 * NB: The values are preserved here to maintain binary compatibility
 * with applications like wpa_supplicant and hostapd.
 */
typedef enum _ieee80211_cipher_type {
    IEEE80211_CIPHER_WEP        = 0,
    IEEE80211_CIPHER_TKIP       = 1,
    IEEE80211_CIPHER_AES_OCB    = 2,
    IEEE80211_CIPHER_AES_CCM    = 3,
    IEEE80211_CIPHER_CKIP       = 5,
    IEEE80211_CIPHER_NONE       = 6,
} ieee80211_cipher_type;

#define IEEE80211_CIPHER_MAX    (IEEE80211_CIPHER_NONE+1)

/* key direction */
typedef enum _ieee80211_key_direction {
    IEEE80211_KEY_DIR_TX,
    IEEE80211_KEY_DIR_RX,
    IEEE80211_KEY_DIR_BOTH
} ieee80211_key_direction;

#define IEEE80211_KEYIX_NONE    ((u_int16_t) -1)

typedef struct _ieee80211_keyval {
    ieee80211_cipher_type   keytype;
    ieee80211_key_direction keydir;
    u_int                   persistent:1,   /* persistent key */
                            mfp:1;          /* management frame protection */
    u_int16_t               keylen;         /* length of the key data fields */
    u_int8_t                *macaddr;       /* mac address of length IEEE80211_ADDR_LEN . all bytes are 0xff for multicast key */
    u_int64_t               keyrsc;
    u_int64_t               keytsc;
    u_int16_t               txmic_offset;   /* TKIP only: offset to tx mic key */
    u_int16_t               rxmic_offset;   /* TKIP only: offset to rx mic key */
    u_int8_t                *keydata;
} ieee80211_keyval;

typedef enum _ieee80211_rsn_param {
    IEEE80211_UCAST_CIPHER_LEN,
    IEEE80211_MCAST_CIPHER_LEN,
    IEEE80211_KEYMGT_ALGS,
    IEEE80211_RSN_CAPS
} ieee80211_rsn_param;

typedef enum _ieee80211_acl_cmd {
    IEEE80211_ACL_ADDMAC,
    IEEE80211_ACL_DELMAC,
    IEEE80211_ACL_POLICY_OPEN,
    IEEE80211_ACL_POLICY_ALLOW,
    IEEE80211_ACL_POLICY_DENY,
    IEEE80211_ACL_POLICY_RADIUS,
    IEEE80211_ACL_FLUSH,
    IEEE80211_ACL_DETACH
} ieee80211_acl_cmd;

#define IEEE80211_PMKID_LEN     16

typedef struct _ieee80211_pmkid_entry {
    u_int8_t    bssid[IEEE80211_ADDR_LEN];
    u_int8_t    pmkid[IEEE80211_PMKID_LEN];
} ieee80211_pmkid_entry;

typedef enum _wlan_wme_param {
    WLAN_WME_CWMIN,
    WLAN_WME_CWMAX,
    WLAN_WME_AIFS,
    WLAN_WME_TXOPLIMIT,
    WLAN_WME_ACM,      /*bss only*/
    WLAN_WME_ACKPOLICY /*bss only*/
} wlan_wme_param;

typedef enum _ieee80211_frame_type {
    IEEE80211_FRAME_TYPE_PROBEREQ,
    IEEE80211_FRAME_TYPE_BEACON,
    IEEE80211_FRAME_TYPE_PROBERESP,
    IEEE80211_FRAME_TYPE_ASSOCREQ,
    IEEE80211_FRAME_TYPE_ASSOCRESP
} ieee80211_frame_type; 

#define IEEE80211_FRAME_TYPE_MAX    (IEEE80211_FRAME_TYPE_ASSOCRESP+1)

typedef enum _ieee80211_ampdu_mode {
    IEEE80211_AMPDU_MODE_OFF,   /* disable AMPDU */
    IEEE80211_AMPDU_MODE_ON,    /* enable AMPDU */
    IEEE80211_AMPDU_MODE_WDSVAR /* enable AMPDU with 4addr WAR */
} ieee80211_ampdu_mode;

typedef enum _ieee80211_reset_type {
    IEEE80211_RESET_TYPE_DEVICE = 0,    /* device reset on error: tx timeout and etc. */
    IEEE80211_RESET_TYPE_DOT11_INTF,    /* dot11 reset: only reset one network interface (vap) */
    IEEE80211_RESET_TYPE_INTERNAL,      /* internal reset */
} ieee80211_reset_type;

typedef struct _ieee80211_reset_request {
    ieee80211_reset_type    type;

    u_int                   reset_hw:1,         /* reset the actual H/W */
                            /*
                             * The following fields are only valid for DOT11 reset, i.e.,
                             * IEEE80211_RESET_TYPE_DOT11_INTF
                             */
                            reset_phy:1,        /* reset PHY */
                            reset_mac:1,        /* reset MAC */
                            set_default_mib:1;  /* set default MIB variables */
    u_int8_t                macaddr[IEEE80211_ADDR_LEN];
} ieee80211_reset_request;

#define IEEE80211_MSG_MLME      0x80000000      /* MLME */
#define IEEE80211_MSG_DEBUG     0x40000000      /* IFF_DEBUG equivalent */
#define IEEE80211_MSG_DUMPPKTS  0x20000000      /* IFF_LINK2 equivalant */
#define IEEE80211_MSG_CRYPTO    0x10000000      /* crypto work */
#define IEEE80211_MSG_INPUT     0x08000000      /* input handling */
#define IEEE80211_MSG_XRATE     0x04000000      /* rate set handling */
#define IEEE80211_MSG_ELEMID    0x02000000      /* element id parsing */
#define IEEE80211_MSG_NODE      0x01000000      /* node handling */
#define IEEE80211_MSG_ASSOC     0x00800000      /* association handling */
#define IEEE80211_MSG_AUTH      0x00400000      /* authentication handling */
#define IEEE80211_MSG_SCAN      0x00200000      /* scanning */
#define IEEE80211_MSG_OUTPUT    0x00100000      /* output handling */
#define IEEE80211_MSG_STATE     0x00080000      /* state machine */
#define IEEE80211_MSG_POWER     0x00040000      /* power save handling */
#define IEEE80211_MSG_DOT1X     0x00020000      /* 802.1x authenticator */
#define IEEE80211_MSG_DOT1XSM   0x00010000      /* 802.1x state machine */
#define IEEE80211_MSG_RADIUS    0x00008000      /* 802.1x radius client */
#define IEEE80211_MSG_RADDUMP   0x00004000      /* dump 802.1x radius packets */
#define IEEE80211_MSG_RADKEYS   0x00002000      /* dump 802.1x keys */
#define IEEE80211_MSG_WPA       0x00001000      /* WPA/RSN protocol */
#define IEEE80211_MSG_ACL       0x00000800      /* ACL handling */
#define IEEE80211_MSG_WME       0x00000400      /* WME protocol */
#define IEEE80211_MSG_SUPG      0x00000200      /* SUPERG */
#define IEEE80211_MSG_DOTH      0x00000100      /* 11.h */
#define IEEE80211_MSG_INACT     0x00000080      /* inactivity handling */
#define IEEE80211_MSG_ROAM      0x00000040      /* sta-mode roaming */
#define IEEE80211_MSG_ACTION    0x00000020      /* action management frames */
#define IEEE80211_MSG_WDS       0x00000010      /* WDS handling */
#define IEEE80211_MSG_ANY       0xffffffff      /* anything */
/*
 * the lower 4 bits of the msg flags are used for extending the
 * debug flags. 
 */ 

/*
 * IEEE80211 PHY Statistics.
 */
struct ieee80211_phy_stats {
    u_int64_t   ips_tx_packets;  /* frames successfully transmitted */
    u_int64_t   ips_tx_multicast; /* multicast/broadcast frames successfully transmitted */
    u_int64_t   ips_tx_fragments; /* fragments successfully transmitted */
    u_int64_t   ips_tx_xretries; /* frames that are xretried. NB: not number of retries */
    u_int64_t   ips_tx_retries;  /* frames transmitted after retries. NB: not number of retries */
    u_int64_t   ips_tx_multiretries; /* frames transmitted after more than one retry. */
    u_int64_t   ips_tx_timeout;  /* frames that expire the dot11MaxTransmitMSDULifetime */
    u_int64_t   ips_rx_packets;  /* frames successfully received */
    u_int64_t   ips_rx_multicast; /* multicast/broadcast frames successfully received */
    u_int64_t   ips_rx_fragments; /* fragments successfully received */
    u_int64_t   ips_rx_timeout;  /* frmaes that expired the dot11MaxReceiveLifetime */
    u_int64_t   ips_rx_dup;      /* duplicated fragments */
    u_int64_t   ips_rx_mdup;     /* multiple duplicated fragments */
    u_int64_t   ips_rx_promiscuous; /* frames that are received only because promiscuous filter is on */
    u_int64_t   ips_rx_promiscuous_fragments; /* fragments that are received only because promiscuous filter is on */
    u_int64_t   ips_tx_rts;         /* RTS success count */
    u_int64_t   ips_tx_shortretry;  /* tx on-chip retries (short). RTSFailCnt */
    u_int64_t   ips_tx_longretry;   /* tx on-chip retries (long). DataFailCnt */
    u_int64_t   ips_rx_crcerr;      /* rx failed 'cuz of bad CRC */
    u_int64_t   ips_rx_fifoerr;     /* rx failed 'cuz of FIFO overrun */
    u_int64_t   ips_rx_decrypterr;  /* rx decryption error */
};

struct ieee80211_mac_stats {
    u_int64_t   ims_tx_packets; /* frames successfully transmitted */
    u_int64_t   ims_rx_packets; /* frames successfully received */

    /* Decryption errors */
    u_int64_t   ims_rx_unencrypted; /* rx w/o wep and privacy on */
    u_int64_t   ims_rx_badkeyid;    /* rx w/ incorrect keyid */
    u_int64_t   ims_rx_decryptok;   /* rx decrypt okay */
    u_int64_t   ims_rx_decryptcrc;  /* rx decrypt failed on crc */
    u_int64_t   ims_rx_wepfail;     /* rx wep processing failed */
    u_int64_t   ims_rx_tkipreplay;  /* rx seq# violation (TKIP) */
    u_int64_t   ims_rx_tkipformat;  /* rx format bad (TKIP) */
    u_int64_t   ims_rx_tkipmic;     /* rx MIC check failed (TKIP) */
    u_int64_t   ims_rx_tkipicv;     /* rx ICV check failed (TKIP) */
    u_int64_t   ims_rx_ccmpreplay;  /* rx seq# violation (CCMP) */
    u_int64_t   ims_rx_ccmpformat;  /* rx format bad (CCMP) */
    u_int64_t   ims_rx_ccmpmic;     /* rx MIC check failed (CCMP) */

    /* Other Tx/Rx errors */
    u_int64_t   ims_tx_discard;     /* tx dropped by NIC */
    u_int64_t   ims_rx_discard;     /* rx dropped by NIC */
};

/*
 * Summary statistics.
 */
struct ieee80211_stats {
	u_int32_t	is_rx_badversion;	/* rx frame with bad version */
	u_int32_t	is_rx_tooshort;		/* rx frame too short */
	u_int32_t	is_rx_wrongbss;		/* rx from wrong bssid */
	u_int32_t	is_rx_wrongdir;		/* rx w/ wrong direction */
	u_int32_t	is_rx_mcastecho;	/* rx discard 'cuz mcast echo */
	u_int32_t	is_rx_notassoc;		/* rx discard 'cuz sta !assoc */
	u_int32_t	is_rx_noprivacy;	/* rx w/ wep but privacy off */
	u_int32_t	is_rx_decap;		/* rx decapsulation failed */
	u_int32_t	is_rx_mgtdiscard;	/* rx discard mgt frames */
	u_int32_t	is_rx_ctl;		/* rx discard ctrl frames */
	u_int32_t	is_rx_beacon;		/* rx beacon frames */
	u_int32_t	is_rx_rstoobig;		/* rx rate set truncated */
	u_int32_t	is_rx_elem_missing;	/* rx required element missing*/
	u_int32_t	is_rx_elem_toobig;	/* rx element too big */
	u_int32_t	is_rx_elem_toosmall;	/* rx element too small */
	u_int32_t	is_rx_elem_unknown;	/* rx element unknown */
	u_int32_t	is_rx_badchan;		/* rx frame w/ invalid chan */
	u_int32_t	is_rx_chanmismatch;	/* rx frame chan mismatch */
	u_int32_t	is_rx_nodealloc;	/* rx frame dropped */
	u_int32_t	is_rx_ssidmismatch;	/* rx frame ssid mismatch  */
	u_int32_t	is_rx_auth_unsupported;	/* rx w/ unsupported auth alg */
	u_int32_t	is_rx_auth_fail;	/* rx sta auth failure */
	u_int32_t	is_rx_auth_countermeasures;/* rx auth discard 'cuz CM */
	u_int32_t	is_rx_assoc_bss;	/* rx assoc from wrong bssid */
	u_int32_t	is_rx_assoc_notauth;	/* rx assoc w/o auth */
	u_int32_t	is_rx_assoc_capmismatch;/* rx assoc w/ cap mismatch */
	u_int32_t	is_rx_assoc_norate;	/* rx assoc w/ no rate match */
	u_int32_t	is_rx_assoc_badwpaie;	/* rx assoc w/ bad WPA IE */
	u_int32_t	is_rx_deauth;		/* rx deauthentication */
	u_int32_t	is_rx_disassoc;		/* rx disassociation */
	u_int32_t	is_rx_action;		/* rx action mgt */
	u_int32_t	is_rx_badsubtype;	/* rx frame w/ unknown subtype*/
	u_int32_t	is_rx_nobuf;		/* rx failed for lack of buf */
	u_int32_t	is_rx_ahdemo_mgt;	/* rx discard ahdemo mgt frame*/
	u_int32_t	is_rx_bad_auth;		/* rx bad auth request */
	u_int32_t	is_rx_unauth;		/* rx on unauthorized port */
	u_int32_t	is_rx_badcipher;	/* rx failed 'cuz key type */
	u_int32_t	is_tx_nodefkey;		/* tx failed 'cuz no defkey */
	u_int32_t	is_tx_noheadroom;	/* tx failed 'cuz no space */
	u_int32_t	is_rx_nocipherctx;	/* rx failed 'cuz key !setup */
	u_int32_t	is_rx_acl;		/* rx discard 'cuz acl policy */
	u_int32_t	is_rx_ffcnt;		/* rx fast frames */
	u_int32_t	is_rx_badathtnl;   	/* driver key alloc failed */
	u_int32_t	is_rx_nowds;     	/* 4-addr packets received with no wds enabled */
	u_int32_t	is_tx_nobuf;		/* tx failed for lack of buf */
	u_int32_t	is_tx_nonode;		/* tx failed for no node */
	u_int32_t	is_tx_unknownmgt;	/* tx of unknown mgt frame */
	u_int32_t	is_tx_badcipher;	/* tx failed 'cuz key type */
	u_int32_t	is_tx_ffokcnt;		/* tx fast frames sent success */
	u_int32_t	is_tx_fferrcnt;		/* tx fast frames sent success */
	u_int32_t	is_scan_active;		/* active scans started */
	u_int32_t	is_scan_passive;	/* passive scans started */
	u_int32_t	is_node_timeout;	/* nodes timed out inactivity */
	u_int32_t	is_crypto_nomem;	/* no memory for crypto ctx */
	u_int32_t	is_crypto_tkip;		/* tkip crypto done in s/w */
	u_int32_t	is_crypto_tkipenmic;	/* tkip en-MIC done in s/w */
	u_int32_t	is_crypto_tkipdemic;	/* tkip de-MIC done in s/w */
	u_int32_t	is_crypto_tkipcm;	/* tkip counter measures */
	u_int32_t	is_crypto_ccmp;		/* ccmp crypto done in s/w */
	u_int32_t	is_crypto_wep;		/* wep crypto done in s/w */
	u_int32_t	is_crypto_setkey_cipher;/* cipher rejected key */
	u_int32_t	is_crypto_setkey_nokey;	/* no key index for setkey */
	u_int32_t	is_crypto_delkey;	/* driver key delete failed */
	u_int32_t	is_crypto_badcipher;	/* unknown cipher */
	u_int32_t	is_crypto_nocipher;	/* cipher not available */
	u_int32_t	is_crypto_attachfail;	/* cipher attach failed */
	u_int32_t	is_crypto_swfallback;	/* cipher fallback to s/w */
	u_int32_t	is_crypto_keyfail;	/* driver key alloc failed */
	u_int32_t	is_crypto_enmicfail;	/* en-MIC failed */
	u_int32_t	is_ibss_capmismatch;	/* merge failed-cap mismatch */
	u_int32_t	is_ibss_norate;		/* merge failed-rate mismatch */
	u_int32_t	is_ps_unassoc;		/* ps-poll for unassoc. sta */
	u_int32_t	is_ps_badaid;		/* ps-poll w/ incorrect aid */
	u_int32_t	is_ps_qempty;		/* ps-poll w/ nothing to send */
};

typedef enum _ieee80211_send_frame_type {
    IEEE80211_SEND_NULL,
    IEEE80211_SEND_QOSNULL,
} ieee80211_send_frame_type;

typedef struct _ieee80211_tspec_info {
    u_int8_t    traffic_type;
    u_int8_t    direction;
    u_int8_t    dot1Dtag;
    u_int8_t    acc_policy_edca;
    u_int8_t    acc_policy_hcca;
    u_int8_t    aggregation;
    u_int8_t    psb;
    u_int8_t    ack_policy;
    u_int16_t   norminal_msdu_size;
    u_int16_t   max_msdu_size;
    u_int32_t   min_srv_interval;
    u_int32_t   max_srv_interval;
    u_int32_t   inactivity_interval;
    u_int32_t   suspension_interval;
    u_int32_t   srv_start_time;
    u_int32_t   min_data_rate;
    u_int32_t   mean_data_rate;
    u_int32_t   peak_data_rate;
    u_int32_t   max_burst_size;
    u_int32_t   delay_bound;
    u_int32_t   min_phy_rate;
    u_int16_t   surplus_bw;
} ieee80211_tspec_info;

#endif
