/*****************************************************************************
**
** cgiMain.c
**
** This is a CGI function that is used to process the tagged HTML files that
** contain the environmental markers.  It also provides a "translation"
** function that will replace tagged parameters with their equivalent values.
**
** Copyright 2008, Atheros Communications, Inc.  All Rights Reserved.
**
******************************************************************************
/*
** include files
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>
#include <stdarg.h>
#include <sys/reboot.h>

/*
** local definitions
*****************
** Local Types
** Structure element for parameter display
*/

typedef struct {
char    *val;
char    *label;
} t_DispLable;

typedef struct {
char    Name[32];
char    Val[70];
} t_singleParam;

#define MAX_WLAN_ELEMENT    1024

typedef struct {
int             numParams;
t_singleParam   Param[MAX_WLAN_ELEMENT];
} EnvParam_t;

/*
** Global Data
*/

extern char **environ;

/*
**  Data
*/


static  EnvParam_t      config;
static  char            rspBuff[65536];
static  char            opBuff[65536];
static  unsigned int    parameterIndex = 0;
static  char            additionalParams[5][32];
static  unsigned int    numAdditionalParams = 0;
static	unsigned		AbortFlag=0;
static  unsigned		ModeFlag=0; // 0 = web page translation, 1 = cmdline

/*
** Internal Prototypes
*/

char *CFG_get_by_name(char *name,char *buff);
char *extractParam(char *str,char *Name,char *Val);
char *Execute_cmd(char *cmd,char *buffer);
char *expandOutput(char *inBuff, char *outBuff);

int  CFG_set_by_name(char *name,char *val);

/******************************************************************************/
/*!
**  \brief Print output to HTML or standard text
**
**  This macro will output a string based either on being in HTML mode
**  or in command line mode.  Used for Debug/Error messages that will show
**  up on the web page or on console output.
**
*/

#define modePrintf(x...) if(ModeFlag){printf(x);printf("\n");} \
						 else \
						 { printf("<p class='header2'>"); \
						   printf(x); printf("</p>"); }

/******************************************************************************/
/*!
**  \brief Fills parameter struct
**
**  This function will fill the parameter data structure with additional
**  parameters as read from either the temp file or the flash file system.
**
**  \param f File pointer to file containing name/value pairs
**  \return N/A
*/

void fillParamStruct(FILE *f)
{
    char            buff[256];
    unsigned long   syncWord;
    
    /*
    ** Read the first word.  It should be 0xfaf30020
    */
    
    fread(&syncWord,4,1,f);
    
    if(syncWord == 0xfaf30020 )
    {
        /*
        ** File has been initialized.  Let's read until we find a NULL
        */
        
        while( !feof(f) )
        {
            /*
            ** Read one line at a time.  Seperated by line feed
            ** characters.
            */
            
            fgets(buff,256,f);
            
            if( buff[0] == 0 )
                break;
                
//            printf("inBuff: %s\n",buff);
            extractParam(buff,config.Param[config.numParams].Name,config.Param[config.numParams++].Val);
        }
    }
}


/*****************************************************************************
**
** processSpecial
**
** This function expands special processing tags.  These tags must exist on a
** single line.  These are used as substution tags in template files that
** are replaced with values in the enviornment or in the saved parameter
** cache.
**
** Formats Supported
**
** ~`executable string`~    Indicates run the command and insert it's output
** ~cParam:Value~           For Checkbox/Radio Button support (if Param = Value)
** ~sParam:Value~           For Select support (if Param=Value)
** ~~Param:Default~     `   Direct Parameter Substution, use default if not defined
** ~?Param:Value`executable`~   Conditional execution, if Param=value
**
**  PARAMETERS:
**
**  RETURNS: 
**
*/

char *processSpecial(char *paramStr, char *outBuff)
{

    char            arg;
    char            *secArg = NULL;
    char            *exeArg = NULL;
    char            *indexPtr;
    char            *exePtr;
    char            paramVal[64];
    char            paramIndexName[48];
    unsigned int    extraParamIndex;
	unsigned int	negate = 0;
	unsigned int	cmpVal;
    
    /*
    ** Get the pointers to the unique components of the parameter string
    */
    
    arg = *paramStr++;
    
    secArg = strchr(paramStr,':');

	/*
	** If a parameter is indicated with a ! instead of a :, then any comparison
	** is negated (used for comparison implementations)
	*/

	if(!secArg)
	{
		secArg = strchr(paramStr,'!');

		if(secArg)
			negate = 1;
	}

	/*
	** If the second argument is specified, break it out
	*/

    if(secArg)
    {
        *secArg++ = 0;
        
        exeArg = strchr(secArg,'`');    
        if(exeArg)
        {
            *exeArg++ = 0;
            exePtr = strchr(exeArg,'`');
            if(exePtr)
                *exePtr = 0;
        }
    }

//    printf("Arg: %c Negate: %d\n",arg,negate);
//    printf("paramStr: %s\n",paramStr);

//    if (secArg)
//        printf("secArg: %s\n",secArg);

//    if (exeArg)
//        printf("exeArg: %s\n",exeArg);
        
    /*
    ** Get the parameter in question
    **
    ** Want to get the name.  If ParamStr contains a # at the end, we will
    ** translate the name with the current index value
    */

    indexPtr = strchr(paramStr,'#');
    if( indexPtr )
    {
        *indexPtr = 0;
        
//        printf("Index Param Str: %s\n",paramStr);
        if( parameterIndex > 1 )
        {
            sprintf(paramIndexName,"%s_%d",paramStr,parameterIndex);
//            printf("%s\n",paramIndexName);
            CFG_get_by_name(paramIndexName,paramVal);
        }
        else
            CFG_get_by_name(paramStr,paramVal);
    }
    else
        CFG_get_by_name(paramStr,paramVal);
   
//    printf("paramVal: %s\n",paramVal);
     
    /*
    ** Unique processing depends on the argument.  The supported formats will
    ** be processed individually
    */
    
    switch (arg)
    {
    case '~':
        /*
        ** Direct Insertion.  If no value, insert default
        */
        
        if( paramVal[0] == 0 && secArg != 0)
            outBuff += sprintf(outBuff,"%s",secArg);
        else
            outBuff += sprintf(outBuff,"%s",paramVal);
        break;

    case '!':
        /*
        ** Abort Line.  If the parameter has no specified OR default
        ** value, simply do not output the line.  Used for file substution
        ** for values that may or may not be there
        */
        
        if( paramVal[0] == 0 && secArg != 0)
            outBuff += sprintf(outBuff,"%s",secArg);
        else
            if( paramVal[0] == 0 )
				AbortFlag = 1;
			else
                outBuff += sprintf(outBuff,"%s",paramVal);
        break;

    case 'c':
        /*
        ** If the sec arg and the value are equal, then put "checked" in the output
        */
        
        if( secArg != NULL)
		{
 		    cmpVal = strcmp(paramVal,secArg);

            if( (negate && cmpVal) || (!negate && !cmpVal) )
                outBuff += sprintf(outBuff,"checked");
		}
        break;
        
    case 's':
        /*
        ** If the sec arg and the value are equal, then put "checked" in the output
        */
        
        if( secArg != NULL)
		{
 		    cmpVal = strcmp(paramVal,secArg);

            if( (negate && cmpVal) || (!negate && !cmpVal) )
                outBuff += sprintf(outBuff,"selected");
		}        
        break;

    case '`':
        /*
        ** Execute the command. Contained in paramStr for this case
        */
        
        exePtr = strchr(paramStr,'`');
        if( exePtr )
            *exePtr = 0;
        
        outBuff = expandOutput(Execute_cmd(paramStr,rspBuff),outBuff);
        
        break;

    case '?':
		cmpVal = strcmp(paramVal,secArg);

		if( (negate && cmpVal) || (!negate && !cmpVal) )
        {
            outBuff = expandOutput(Execute_cmd(exeArg,rspBuff),outBuff);
        }
 
        break;
    
    case '$':
        /*
        ** Insert "extra" Parameter by index
        */
        
        extraParamIndex = atoi(paramStr) - 1;
        if(extraParamIndex < numAdditionalParams)
            outBuff += sprintf(outBuff,"%s",additionalParams[extraParamIndex]);        
            
        break;
    
    case '#':
        /*
        ** Insert _Index to allow for indexed parameter names
        */
        
        if(parameterIndex > 1)
            outBuff += sprintf(outBuff,"_%d",parameterIndex);        
        break;
    
    case 'e':
        /*
        ** Enable the line.  This is used in cases where the parameter line in a file
        ** is dependant on another variable.  If it's enabled, then further processing
        ** can occur.  If not, then the line is abandoned.
        */

        cmpVal = strcmp(paramVal,secArg);
		if( (!negate && cmpVal) || (negate && !cmpVal) )
           	AbortFlag = 1;

		break;
    }

    return outBuff;
}

/*****************************************************************************
**
** expandOutput
**
** This function checks the input buffer, and replaces all cr/lf or lf
** strings with <br> strings.  Used to "html-ify" output data from an
** embedded command.
**
**  PARAMETERS:
**
**  RETURNS: 
**
*/

char *expandOutput(char *inBuff, char *outBuff)
{
    int wasSpace = 0;

    /*
    ** Go until the line has a NULL
    */
    
    while(*inBuff)
    {
        if ( *inBuff == 0x0a)
        {
            wasSpace = 0;
            inBuff++;
            strcpy(outBuff,"<br>");
            outBuff+=4;
        }
        else if ( *inBuff == 0x0d )
        {
            wasSpace = 0;
            inBuff++;
        }
        else if ( *inBuff == ' ' )
        {
            if(wasSpace)
            {
                strcpy(outBuff,"&nbsp;");
                outBuff+=5;
            }
            else
                wasSpace = 1;

            inBuff++;
        }
        else if ( *inBuff == 0x08 )
        {
            wasSpace = 0;
            strcpy(outBuff,"&nbsp;&nbsp;&nbsp;&nbsp;");
            outBuff+=20;
            inBuff++;
        }
        else
        {
            wasSpace = 0;
            *outBuff++ = *inBuff++;
        }
    }
    
    return outBuff;
}


/*****************************************************************************
**
** expandLine
**
** This function checks the input provided, and expands any section that is
** market with the special ~ sequences.  These sequences indicate a specific
** action to be taken regarding the parameter that
** is marked with the ~param~ marker.  It returns the a line that has been
** updated with the proper strings.
**
**  PARAMETERS:
**
**  RETURNS: 
**
*/

char *expandLine(char *line,char *outBuff)
{
    int     respLen;
    char    paramStr[64];
    char    *p;

    /*
    ** Go until the line has a LF or NULL
    */
    
    while(*line)
    {
        if ( *line == 0x0a || *line == 0x0d)
        {
            *outBuff++ = *line++;
            break;
        }

        if ( *line == '~' )
        {
            /*
            ** This is a special parameter.  The parameter string
            ** will be copied into a temporary string, and passed to
            ** the special parameter processing function.
            */
            
            p = paramStr;
            line++;
            *p++ =*line++;    /* this is the qualifier character, "always there" */
            
            while(*line != '~')
                *p++ = *line++;
                
            line++; /* Increment past the last ~ */
            *p = 0; /* Null Terimnate, and line now points at "after" the parameter string */
            
            /*
            ** At this point paramStr contains the full parameter string, ready
            ** for expansion
            */
            
            outBuff = processSpecial(paramStr,outBuff);

			/*
			** If an abort flag is raised, return now
			*/

			if( AbortFlag)
				return (NULL);
        }
        else
        {
            *outBuff++ = *line++;
        }
    }
    
    *outBuff = 0;
    return NULL;
}

/*****************************************************************************
**
** Execute_cmd
**
** This function executes the given command, and returns the results in the
** buffer provided.  Usually good for one line response commands
**
**  RETURNS:
**      Output Buffer
**
*/

char *Execute_cmd(char *cmd,char *buffer)
{
    FILE            *f;
    char            *retBuff = buffer;

    /*
    ** Code Begins
    ** Create the command string to give to popen
    */

    f = popen(cmd, "r");

    if(f)
    {
        /*
        ** Read each line.
        */

        while(1)
        {
            *buffer = 0;
            fgets(buffer,120,f);
            //printf("IN:%s\n",buffer);

            if(strlen(buffer) == 0)
            {
                //printf("END\n");
                break;
            }
                
            //printf("String Length: %d\n",strlen(buffer));
            
            strcat(buffer,"<br>");
            buffer += strlen(buffer);
        }

        pclose(f);
    }

    return(retBuff);
}

/*****************************************************************************
**
** setParamValue
**
** This function puts a parameter value into the indicated location, processing
** for %i or %s markers that require the device ID or serial number to be
** inserted.
**
**  PARAMETERS:
**
**  RETURNS: 
**
*/

void setParamValue(char *targ,char *val)
{
    int     index = 0;

    /*
    ** Code begins.
    ** Assume the value is null terminated
    */

    while(*val)
    {
        if(*val == 0x0a || *val == 0x0d)
        {
            /*
            ** line feed or carrage return.  This should be truncated, end of string
            */
            
            break;
        }
        else
            targ[index++] = *val;

        val++;
    }
    
    targ[index] = 0;    // Insert null terminator
}

/*****************************************************************************
**
** CheckValue
**
** Performs input validation on fields that must conform to certain values.
**
**  PARAMETERS:
**
**  RETURNS: 
**
*/


int CheckValue(char *name,char *val)
{
    int     len;
    int     i;

    /*
    ** Code Begins
    ** Check for the fields of interest
    **
    ** SSID.  Must be 2 to 32 characters long, certain characters
    ** now allowed.
    */

    if(!strncmp(name,"AP_SSID",6))
    {
        char    *badVals = "?\"\\#[]+ ";

        len = strlen(val);

        if(len < 2 || len > 32)
        {
            modePrintf("%s=%s invalid lendth (length %d != 2 to 32 chars)",name,val,len);
            return (-1);
        }
        
        for(i=0;i<strlen(badVals);i++)
        {
            if(strchr(val,badVals[i]))
            {
                modePrintf("%s=%s has invalid character '%c'",name,val,badVals[i]);
                return (-1);
            }
        }
        
        if(val[0] == '!' || val[0] == ';')
        {
            modePrintf("%s=%s invalid first character (cannot be ! or ; )",name,val);
            return (-1);
        }
    }

	/*
	** PSK Key
	** If Defined, it must conform to the 8 to 64 character rule
	*/

    if(!strncmp(name,"PSK_KEY",7) && (strlen(name) < 10))
    {
        len = strlen(val);

        if((len < 8 || len > 63) && len != 0)
        {
            modePrintf("%s=%s invalid lendth (length %d != 8 to 63 chars)",name,val,len);
            return (-1);
        }        
    }

	/*
	** PSK_KEY_HEX
	** This value is the hex version of the PSK KEY.  Must be a hex digit, and
	** be 64 characters long
	*/

    if(!strncmp(name,"PSK_KEY_HEX",11) && (strlen(name) > 10))
    {
        len = strlen(val);

        if(len != 64 && len != 0)
        {
            printf("%s=%s invalid lendth (length %d != 64 chars)",name,val,len);
            return (-1);
        }
        
        for(i=0;i<64;i++)
        {
			if(!isxdigit(val[i]))
			{
				modePrintf("%s=%s has non hex digit, digit='%c' pos=%d",name,val,val[i],i);
                return (-1);
            }
        }        
    }

	/*
	** WEP_KEY
	** We need to check the validity of WEP keys.  If the first two characters
	** of the key are s:, then the length must correspond to a 5 character or 13
	** character key.  If the first two characters are not s:, then it's a hex
	** key and should be a 10 or 26 character hex value.
	*/

    if(!strncmp(name,"WEPKEY",6))
    {
		if((val[0] == 's' || val[0] == 'S') && val[1] == ':')
		{
            char	*valstr = &val[2];

			len = strlen(valstr);

			if(len != 0 && len != 5 && len != 13)
			{
                modePrintf(" %s=%s invalid WEP Key length (length %d != 5 or 13 chars)",
                           name,valstr,len);
                return (-1);
			}
		}
		else
		{
        	len = strlen(val);

        	if((len != 10) && (len != 26) && (len != 32)  && (len != 0))
			{
                modePrintf("%s=%s invalid HEX WEP Key length (length %d != 10 or 26 digits)",name,val,len);
                return (-1);
			}
        
        	for(i=0;i<len;i++)
        	{
				if(!isxdigit(val[i]))
				{
                    modePrintf("%s=%s has non hex digit, digit='%c' pos=%d",name,val,val[i],i);
        	        return (-1);
        	    }
        	}        
		}
    }

    if(!strncmp(name,"AP_VLAN",7))
    {
        if(val)
        {
            int tagval = atoi(val);

            /* when posted through WEB, we get this parameter as zero */
            if (!tagval) return 0;

            /* tag should be between 2-4094. Linux uses 4095 for its
             * internal use 
             */
            if((tagval >= 2 && tagval <= 4094))
            {
                return 0;
            }
            else
            {
                modePrintf("invalid vlan tag value, 2 <= vlan <= 4094");
                return -1;
            }
        }
    }
    return (0); // Parameter is OK.
}


/*****************************************************************************
**
** /brief writes parameters to file
**
** This function will write the save parameter list to the file pointer
** provided.  It is assumed that the file pointer is open and positioned to
** the correct location.  File opening and closing is the responsibility of
** the caller.
*/

void writeParameters(char *name,char *mode,unsigned long offset)
{
    int         i;
    FILE        *f;
    u_int32_t   Sync = 0xfaf30020;
    u_int32_t   Zero = 0;
    
    /*
    ** Code Begins
    ** The name of the file, and the offset into the file are passed as parameters.
    ** This will return an error (-1) if the file does not exist
    */

    f = fopen(name,mode);
    
    if(f)
    {
        /*
        ** If an offset is provided, seek to the offset
        */
            
        if(offset != 0)
            fseek( f, offset, SEEK_SET);
            
        /*
        ** Start writing the file.  Write sync word, then parameters
        */
    
        fwrite(&Sync,4,1,f);

        for(i=0;i<config.numParams;i++)
        {
            /*
            ** We don't want to store the "update" or "commit" parameters, so
            ** remove them if we get here.  Also, if we have values that have
            ** no value, don't write them out.
            */
        
            if( !strcmp(config.Param[i].Name,"UPDATE") )
                continue;
            if( !strcmp(config.Param[i].Name,"COMMIT") )
                continue;
            if( config.Param[i].Val[0] == 0)
                continue;
            
            fprintf(f,"%s=%s\n",config.Param[i].Name,config.Param[i].Val);
        }

        fwrite(&Zero,4,1,f);
        
        fclose(f);
    }
}

/*****************************************************************************
**
** CFG_set_by_name
**
** This will set the parameter specified by name to the indicated value.  Values
** are always strings.
**
**  PARAMETERS:
**          name        Name of the parameter to modify
**          val         New value for the parameter
**
**  RETURNS: 
**          0 on success, -1 on failure
**
*/

int CFG_set_by_name(char *name,char *val)
{
    int     i;
    
    if( CheckValue(name, val) )
        return (-1);

    for( i=0; i < config.numParams; i++ )
    {
        if( !strcmp(config.Param[i].Name,name) )
        {
            /*
            ** This is der one.
            */

            setParamValue(config.Param[i].Val, val);
            return (0);     // Done
        }
    }

    /*
    ** If we get here, we did not find the item.  Insert as a new one
    */

    if(config.numParams < MAX_WLAN_ELEMENT)
    {
        strcpy(config.Param[config.numParams].Name,name);
        setParamValue(config.Param[config.numParams++].Val,val);
    }

	return (0);
}

/*****************************************************************************
**
** CFG_get_by_name
**
** This function gets the parameters from the config structure
**
**  PARAMETERS:
**
**  RETURNS: 
**
*/

char *CFG_get_by_name(char *name,char *buff)
{
    int     i;

    *buff = 0;  // null terminate
    
    for(i=0;i<config.numParams;i++)
    {
        if(!strcmp(config.Param[i].Name,name))
        {
            /*
            ** This is der one.  Find out if there is a %s or %i
            ** in the stream.  If so, insert the proper value
            */
            
            strcpy(buff,config.Param[i].Val);
            break;
        }
    }

    return buff;     // Done
}

/*****************************************************************************
**
** CFG_remove_by_name
**
** This function removes a parameter from the config structure
**
**  PARAMETERS:
**
**  RETURNS: 
**
*/

void CFG_remove_by_name(char *name)
{
    int     i;

    
    for(i=0;i<config.numParams;i++)
    {
        if(!strcmp(config.Param[i].Name,name))
        {
            /*
            ** This is the one.  Move the rest of the items on the list
            ** "up" by one, and decrement the total number of
            ** parameters
            */
            
            for(i++;i<config.numParams;i++)
            {
                strcpy(config.Param[i-1].Name,config.Param[i].Name);
                strcpy(config.Param[i-1].Val,config.Param[i].Val);
            }
            config.numParams--;
            return;
        }
    }
}




/*****************************************************************************
**
** extractVal
**
** This function returns both the value name and value string for the next item in
** the list.  It returns a pointer to the next value, or NULL if the string has
** "run out".
**
**  PARAMETERS:
**
**  RETURNS:
**
*/

char *extractParam(char *str,char *Name,char *Val)
{
    int     param=0;
    int     val = 0;
    
    /*
    ** Code Begins
    ** Search for the ? or & to start the string
    */
    
    while(*str)
    {
        /*
        ** Check for the beginning ? or &.  This signifies the start or
        ** end of the parameter. start is null at the start
        */
 
// 		printf("STRVAL: %x\n",*str);       
        if(*str == '?' || *str=='&' || *str == 0x0a || *str == 0x0d )
        {
            if(!param)
            {
                param = 1;
            }
            else
            {
                /*
                ** All Done.  Return this pointer
                */

                *Val = 0;                
                return (str);
            }
        }
        else if(*str == '=')
        {
            val = 1;
            *Name = 0;  // Null terminate
        }
        else if(!val)
        {
            param = 1;
            *Name++ = *str;
        }
        else
            *Val++ = *str;
            
        str++;
    }
    
    /*
    ** If we get here, we have run out before getting a complete
    ** parameter. End of the line
    */
    
    return (NULL);
}

void unencode(char *src, char *dest)
{
    for(; *src != '\0'; src++)
    {
        if(*src == '+') 
        {
            *dest = ' ';
            dest++;
        }
        //decoding special symbols
        else if(*src == '%') 
        {
            int code;
            if(sscanf(src+1, "%2x", &code) != 1) code = '?';
            // ignoring all newline symbols that come from form - we put our own instead of "&varname1=" --> '\n'
            if(code != 10 && code != 12 && code != 13)
            {
                *dest = code;
                dest++;
            }
            src +=2;
        }
        else
        {
            *dest = *src;
            dest++;
        }
    } //outer for loop

    *dest = 0;
}

/*****************************************************************************
**
** /brief translateFile
**
** This function will read a provided file name, and output the file with
** any substutions included.  This is used to translate template files into
** specific files that have required parameters included.
**
** An optional "index" variable will be used to look for parameters that
** have a specific index (such as AP_SSID_2, etc).  If no index is specified,
** then the parameter is assumed to be not there.  If the index is specified,
** then parameters with a tailing "_#" will have # replaced with the parameter
** ID.
**
*/

int translateFile(char *fname)
{
    char            Name[32];
    char            Value[64];
    char            line[1024];
    FILE            *f;

    /*
    ** Code Begins.
    ** Input the parameter cache for processing
    */

    f = fopen(fname,"r");
    
    if ( !f )
    {
        return (-1);
    }
    
    /*
    ** Read the file, one line at a time.  If the line is aborted, then
	** dump the line and continue
    */
    
    while(!feof(f))
    {
		line[0] = 0;
        fgets(line,1024,f);
        expandLine(line,opBuff);

		if( !AbortFlag )
            printf("%s",opBuff);
		else
			AbortFlag = 0;

		opBuff[0] = 0;	// clear the buffer for the next cycle

    }
    
    fclose ( f );
    
    return (0);
}


/*****************************************************************************
**
** /brief Main
**
** This program will read parameterized HTML files and insert the proper
** strings from the parameter store -- either in flash or temp storage.
**
** The specific page to process will be determined by the actual command
** name returned in argv[0].  Each page name will be linked to the executable
** in much the same manner as busybox does it's commands.  This will require
** that each page link be located in the cgi-bin directory since they are ALL
** being processed as cgi scripts by the httpd daemon.  Actual pages will be
** located in the /usr/www directory.
**
** Other functions are provided to support the command line processing.
**
** Options: -a  Add a parameter/value pair to the cache
**              # cgiMain -a SSID=MySSID
**
**          -r  Remove a parameter from the parameter cache
**              # cgiMain -r AP_THIS_PARAM
**
**          -c  Commit parameter cache to flash
**              # cgiMain -c
**
**          -e  Print the export list for use in scripts
**              `cgiMain -e`
**
**          -i  Invalidate the parameter cache by re-reading the flash
**              values and overriding the parameter cache.  NOTE: this
**              will loose any changes made to the parameter cache.
**              # cgiMain -i
**
**          -t  Translate File.  This will take the indicated file and
**              insert parameter data as marked by the ~ markers in the
**              file.  Uses the same rules as the HTML files.  Normal
**              output is to stdout, can be redirected to another file.
**              if the third parameter is specified, it is assumed to be
**              the interface name.  
**
**              # cgiMain -t wpapsk.conf ath0 > /tmp/secvap.conf
**              # cgiMain -t2 wpapsk.conf > /tmp/secvap2.conf
*/

int main(int argc,char **argv)
{
    char            Page[64];
    char            Name[32];
    char            Value[70];
    char            valBuff[128];
    char            *outPtr;
    int             i;
    int             j;
    int             holdOutput;
    unsigned long   syncWord;
    
    char            *nextField;
    char            *update;
    FILE            *f;

    /*
    ** Code Begins.
    ** Zero out the config structure, and read the parameter cache
    ** (or flash, depending on command)
    */

    memset(&config,0,sizeof(config));

    f = fopen("/tmp/.apcfg","r");
    
    if ( !f )
    {
        /*
        ** We need to read the parameter data starting at 32K into the calibration
        ** sector.  This is mapped to /dev/caldata, so we simply open that device and
        ** read until we hit the EOL
        */
        
        f = fopen( "/dev/caldata", "r" );

		if (!f)
		{
			printf("ERROR:  /dev/caldata not defined on this device\n");
			printf("ERROR:  Cannot store data in flash!!!!\n");
			exit(-1);
		}

        fseek( f, 32 * 1024, SEEK_SET);
    }
    
    /*
    ** At this point the file is either open or not.  If it is, read the 
    ** parameters as require
    */
    
    if ( f )
    {
//        printf("Filling Parameters<br>");
        fillParamStruct(f);
        fclose(f);
//        printf("Total Parameters: %d<br>",config.numParams);
    }
    
    /*
    ** Now we look for options.
    ** -t means translate the indicated file
    */
    
    if(argc > 1)
    {
        if(!strncmp(argv[1],"-t",2))
        {
            /*
            ** Translate file
            ** read the file, then output the translated version
            */
            parameterIndex = 0;
            
            if(argv[1][2] != 0)
                parameterIndex = argv[1][2] - 0x30;
                
            if(parameterIndex > 8)
                parameterIndex = 0;
                
            /*
            ** Input the "extra" parameters that may be included
            */
            
            for(i=3,numAdditionalParams=0;i<argc;i++)
            {
                strcpy(additionalParams[numAdditionalParams++],argv[i]);
            }
            
            /*
            ** Now, perform the translation
            */
                
            translateFile(argv[2]);
            exit(0);
        } 
        else if(!strncmp(argv[1],"-a",2))
        {
            /*
            ** Add a parameter.  Argv[2] should contain the parameter=value string.
            ** Need to add the "&" to the string to make it look like a parameter
            ** string.
            */
            
            sprintf(valBuff,"&%s&",argv[2]);
            extractParam(valBuff,Name,Value);

			ModeFlag = 1;

			/*
			** If setting fails, return a -1 (for scripts)
			*/

			if( CFG_set_by_name(Name,Value) )
				exit(-1);

            writeParameters("/tmp/.apcfg","w+",0);
            exit(0);
        }
        else if(!strncmp(argv[1],"-c",2))
        {
            /*
            ** Write the parameter structure to the flash.  This is
            ** the "commit" function
            */
            
            writeParameters("/dev/caldata","w+",32 * 1024);
            writeParameters("/tmp/.apcfg","w+",0);
            exit(0);
        }
        else if(!strncmp(argv[1],"-r",2))
        {
            /*
            ** Remove the parameter from the cache.  This will write the
            ** cache, but not write to the flash.  Explicit commit required
            ** to write to flash.
            */
            
            CFG_remove_by_name(argv[2]);
            writeParameters("/tmp/.apcfg","w+",0);
            exit(0);
        }
        else if(!strncmp(argv[1],"-e",2))
        {
            /*
            ** Export the variables
            ** This is used as part of a shell script to "export" the variables
            ** to the environment
            */
            for(i=0;i<config.numParams;i++)
            {
                printf("export %s=%s\n",config.Param[i].Name,config.Param[i].Val);
            }
            exit(0);
        }
        else if(!strncmp(argv[1],"-x",2))
        {
            /*
            ** Erase all parameters in flash and cache.
            ** This is the equivalent of a reset command.
            */

            memset(&config,0,sizeof(config));
            writeParameters("/dev/caldata","w+",32 * 1024);
            writeParameters("/tmp/.apcfg","w+",0);
            exit(0);
        }
    }
    
    /*
    ** Otherwise, this is processing for an HTML page
    ** Zero out the config structure, and get the page name
    */
    
    strcpy(Page,"../");
    strcat(Page,argv[0]);
    strcat(Page,".html");
    
    /*
    ** Open the temp file, if it exists.  If not, open the main
    ** flash parameter store
    */

//    printf("Content-Type:text/html\n\n");
//    printf("<HTML><HEAD>\r\n");
//    printf("<LINK REL=\"stylesheet\" href=\"styleSheet.css\" type=\"text/css\">");
//    printf("</head><body>");
  
//    for(i=0;i<argc;i++)
//        printf("Args[%d]=%s<br>",i,argv[i]);
        
//    i=0;
//    while(environ[i] != NULL)
//        printf("ENV: %s<br>\n",environ[i++]);
     
    
    /*
    ** Now to get the environment data.
    ** We parse the input until all parameters are inserted.  If we see a reset, commit,
    ** or accept/submit label, we do the appropriate action
    */
    /* if every thing fine, make sure we add 200 OK with proper content type.
     * At this point we do not know Content-Length, but http 1.0 restricts to
     * add content length so just tell the browser read until connection close
     */
    printf("HTTP/1.0 200 OK\r\n");
    printf("Content-type: text/html\r\n");
    printf("Connection: close\r\n");
    printf("\r\n");
    printf("\r\n");
    nextField = getenv("CONTENT_LENGTH");
    
    if (nextField == NULL)
    {
        sprintf(valBuff,"?%s",getenv("QUERY_STRING"));
        nextField = valBuff;
    }

//    printf("next: %s<br>\n",nextField);

    if(nextField != NULL)
    {
        if(*nextField != '?')
        {
            j = atoi(nextField);

            memset(opBuff,0,1024);
            fgets(opBuff,j+3,stdin);
            nextField = opBuff;
        }

        /*
        ** Check for the reboot button
        ** If hit, we die gloriously
        */
        
        update = strstr(nextField,"RebootButton");
        if(update)
        {
            reboot(RB_AUTOBOOT);
        }

        /*
        ** We want to read all parameters regardless, and update
        ** what was read from tmp/flash.  If the commit parameter
        ** is set, we will write to flash
        */

        while(nextField)
        {
            nextField = extractParam(nextField,Name,valBuff);
            unencode(valBuff,Value);
            
            if(!strcmp("INDEX",Name))
            {
               parameterIndex = atoi(Value);
			   sprintf(Value,"%d",parameterIndex);
            }
            CFG_set_by_name(Name,Value);
             
//            modePrintf("Name: %s Value: %s",Name,Value);
        }
        
        /*
        ** Now, look for the update and/or commit strings to send to either
        ** the temp file or the flash file
        */
        
        if(strcmp(CFG_get_by_name("COMMIT",valBuff),"Commit") == 0 )
        {
            writeParameters("/dev/caldata","w+",32 * 1024);
            writeParameters("/tmp/.apcfg","w+",0);
        }
        
        if(strcmp(CFG_get_by_name("UPDATE",valBuff),"Update") == 0 )
        {
            writeParameters("/tmp/.apcfg","w+",0);
        }

        if(strcmp(CFG_get_by_name("StopButton",valBuff),"Stop") == 0 )
        {
            Execute_cmd("apdown 2>/dev/null", rspBuff);
        }

        if(strcmp(CFG_get_by_name("StartButton",valBuff),"Start") == 0 )
        {
            Execute_cmd("apup 2>/dev/null", rspBuff);
        }
        if(strcmp(CFG_get_by_name("FactoryResetButton",valBuff),"FactoryReset") == 0 )
        {
            Execute_cmd("cfg -x 2>/dev/null;/etc/ath/apcfg", rspBuff);

			/*
			** Now, we need to re-read the parameters from the cache to get
			** what the apcfg has reconfigured for this iteration of the page
			*/

            f = fopen("/tmp/.apcfg","r");
    
		    if ( f )
    		{
				/*
				** Zero out the parameter structure and re-read
				*/
                memset(&config,0,sizeof(config));
        		fillParamStruct(f);
        		fclose(f);
    		}
        }
        if(strcmp(CFG_get_by_name("StartPINMethod",valBuff),"StartPINMethod") == 0 )
        {
            /* extract the enrollee key and pass it over to the command 
             */
            char cmd[256]={0};

            sprintf(cmd, "wpatalk -v ath%c 'configthem pin=%s'", '0'+(parameterIndex-1), CFG_get_by_name("AP_ENROLLEE", valBuff));
            Execute_cmd(cmd, rspBuff);
        }
        if(strcmp(CFG_get_by_name("StartPBC",valBuff),"StartPBC") == 0 )
        {
            /* extract the enrollee key and pass it over to the command 
             */
            char cmd[256]={0};
            sprintf(cmd, "wpatalk -v ath0 configthem");
            Execute_cmd(cmd, rspBuff);
        }
    }

    /*
    ** use the "translate file" function to translate the file, inserting the
    ** special strings as required.
    */
    
    if(translateFile(Page) < 0)
    {
        printf("Content-Type:text/html\n\n");
        printf("<HTML><HEAD>\r\n");
        printf("<LINK REL=\"stylesheet\" href=\"../styleSheet.css\" type=\"text/css\">");
        printf("</head><body>");
        printf("Page %s Not Found",Page);
        printf("</body></html>");
        exit(1);
    }
    
    return (0);
}

/********************************** End of Module *****************************/

